from fnmatch import fnmatch
from heuristics.heuristic_base import Heuristic

def get_parts(fact):
    """Extract the components of a PDDL fact by removing parentheses and splitting the string."""
    return fact[1:-1].split()

def match(fact, *args):
    """
    Check if a PDDL fact matches a given pattern.

    - `fact`: The complete fact as a string, e.g., "(at bob shed)".
    - `args`: The expected pattern (wildcards `*` allowed).
    - Returns `True` if the fact matches the pattern, else `False`.
    """
    parts = get_parts(fact)
    return all(fnmatch(part, arg) for part, arg in zip(parts, args))

class SpannerHeuristic(Heuristic):
    """
    A domain-dependent heuristic for the Spanner domain.

    # Summary
    This heuristic estimates the number of actions needed to tighten all loose nuts in the Spanner domain.
    It considers the following:
    - The man must pick up a usable spanner if not already carrying one.
    - The man must walk to the location of each loose nut.
    - The man must tighten each loose nut using a usable spanner.

    # Assumptions:
    - The man can carry only one spanner at a time.
    - A spanner becomes unusable after tightening a nut.
    - The man must walk to the location of each nut to tighten it.

    # Heuristic Initialization
    - Extract the goal conditions (tightened nuts) and static facts (links between locations).
    - Build a graph of locations using the static `link` facts to compute distances.

    # Step-By-Step Thinking for Computing Heuristic
    1. Identify the number of loose nuts that need to be tightened.
    2. Check if the man is carrying a usable spanner:
       - If not, add the cost of picking up a usable spanner.
    3. For each loose nut:
       - Compute the shortest path from the man's current location to the nut's location.
       - Add the cost of walking along this path.
       - Add the cost of tightening the nut (1 action).
    4. Sum the total cost of all actions required to tighten all loose nuts.
    """

    def __init__(self, task):
        """Initialize the heuristic by extracting goal conditions and static facts."""
        self.goals = task.goals  # Goal conditions.
        static_facts = task.static  # Static facts (e.g., links between locations).

        # Build a graph of locations using the static `link` facts.
        self.location_graph = {}
        for fact in static_facts:
            if match(fact, "link", "*", "*"):
                _, loc1, loc2 = get_parts(fact)
                if loc1 not in self.location_graph:
                    self.location_graph[loc1] = []
                if loc2 not in self.location_graph:
                    self.location_graph[loc2] = []
                self.location_graph[loc1].append(loc2)
                self.location_graph[loc2].append(loc1)

    def __call__(self, node):
        """Estimate the minimum cost to tighten all loose nuts."""
        state = node.state

        # Count the number of loose nuts that need to be tightened.
        loose_nuts = sum(1 for fact in state if match(fact, "loose", "*"))

        # Check if the man is carrying a usable spanner.
        carrying_spanner = any(match(fact, "carrying", "*", "*") for fact in state)
        usable_spanner = any(match(fact, "usable", "*") for fact in state)

        # Initialize the heuristic cost.
        total_cost = 0

        # If the man is not carrying a usable spanner, add the cost of picking one up.
        if not (carrying_spanner and usable_spanner):
            total_cost += 1  # Cost of picking up a spanner.

        # For each loose nut, compute the cost of walking to its location and tightening it.
        for fact in state:
            if match(fact, "loose", "*"):
                _, nut = get_parts(fact)
                # Find the location of the nut.
                nut_location = None
                for loc_fact in state:
                    if match(loc_fact, "at", nut, "*"):
                        _, _, nut_location = get_parts(loc_fact)
                        break
                # Find the man's current location.
                man_location = None
                for loc_fact in state:
                    if match(loc_fact, "at", "*", "*") and not match(loc_fact, "at", "spanner*", "*"):
                        _, man, man_location = get_parts(loc_fact)
                        break
                # Compute the shortest path from the man's location to the nut's location.
                if man_location and nut_location:
                    path_length = self._shortest_path_length(man_location, nut_location)
                    total_cost += path_length  # Cost of walking.
                total_cost += 1  # Cost of tightening the nut.

        return total_cost

    def _shortest_path_length(self, start, end):
        """
        Compute the shortest path length between two locations using BFS.

        - `start`: The starting location.
        - `end`: The target location.
        - Returns the number of steps in the shortest path.
        """
        if start == end:
            return 0
        visited = set()
        queue = [(start, 0)]
        while queue:
            current, steps = queue.pop(0)
            if current == end:
                return steps
            if current not in visited:
                visited.add(current)
                for neighbor in self.location_graph.get(current, []):
                    queue.append((neighbor, steps + 1))
        return float('inf')  # If no path exists (should not happen in valid instances).
