from fnmatch import fnmatch
from heuristics.heuristic_base import Heuristic

def get_parts(fact):
    """Extract the components of a PDDL fact by removing parentheses and splitting the string."""
    return fact[1:-1].split()

def match(fact, *args):
    """
    Check if a PDDL fact matches a given pattern.

    - `fact`: The complete fact as a string, e.g., "(at bob shed)".
    - `args`: The expected pattern (wildcards `*` allowed).
    - Returns `True` if the fact matches the pattern, else `False`.
    """
    parts = get_parts(fact)
    return all(fnmatch(part, arg) for part, arg in zip(parts, args))

class SpannerHeuristic(Heuristic):
    """
    A domain-dependent heuristic for the Spanner domain.

    # Summary
    This heuristic estimates the number of actions needed to tighten all loose nuts in the Spanner domain.
    It considers the following:
    - The man must pick up a usable spanner if not already carrying one.
    - The man must walk to the location of each loose nut.
    - The man must tighten each loose nut using a spanner.

    # Assumptions:
    - The man can carry only one spanner at a time.
    - A spanner can be used only once (it becomes unusable after tightening a nut).
    - The man must walk to the location of a spanner to pick it up.
    - The man must walk to the location of a nut to tighten it.

    # Heuristic Initialization
    - Extract the goal conditions (tightened nuts) and static facts (links between locations).
    - Create a mapping of locations to their connected locations using the `link` facts.

    # Step-By-Step Thinking for Computing Heuristic
    1. Identify the number of loose nuts that need to be tightened.
    2. Check if the man is carrying a usable spanner:
       - If not, add the cost of picking up a spanner (walk to the spanner's location and pick it up).
    3. For each loose nut:
       - Add the cost of walking to the nut's location.
       - Add the cost of tightening the nut (if a usable spanner is available).
    4. If the man is not at the starting location of the walk, add the cost of walking to the starting location.
    """

    def __init__(self, task):
        """Initialize the heuristic by extracting goal conditions and static facts."""
        self.goals = task.goals  # Goal conditions.
        static_facts = task.static  # Facts that are not affected by actions.

        # Create a mapping of locations to their connected locations using "link" facts.
        self.location_links = {}
        for fact in static_facts:
            if match(fact, "link", "*", "*"):
                _, loc1, loc2 = get_parts(fact)
                if loc1 not in self.location_links:
                    self.location_links[loc1] = set()
                if loc2 not in self.location_links:
                    self.location_links[loc2] = set()
                self.location_links[loc1].add(loc2)
                self.location_links[loc2].add(loc1)

    def __call__(self, node):
        """Estimate the minimum cost to tighten all loose nuts."""
        state = node.state

        # Count the number of loose nuts that need to be tightened.
        loose_nuts = sum(1 for fact in state if match(fact, "loose", "*"))

        # Check if the man is carrying a usable spanner.
        carrying_spanner = any(match(fact, "carrying", "*", "*") for fact in state)
        usable_spanner = any(match(fact, "usable", "*") for fact in state)

        # Initialize the heuristic cost.
        total_cost = 0

        # If the man is not carrying a usable spanner, add the cost of picking one up.
        if not (carrying_spanner and usable_spanner):
            # Find the closest spanner to the man's current location.
            man_location = None
            for fact in state:
                if match(fact, "at", "*", "*"):
                    _, obj, loc = get_parts(fact)
                    if obj == "bob":
                        man_location = loc
                        break

            if man_location:
                # Find the closest spanner location.
                spanner_locations = [get_parts(fact)[2] for fact in state if match(fact, "at", "spanner*", "*")]
                if spanner_locations:
                    # Assume the cost of walking to the closest spanner is 1 action.
                    total_cost += 1  # Walk to the spanner.
                    total_cost += 1  # Pick up the spanner.

        # Add the cost of walking to each loose nut and tightening it.
        total_cost += loose_nuts * 2  # Walk to the nut and tighten it.

        return total_cost
