from fnmatch import fnmatch
from heuristics.heuristic_base import Heuristic

def get_parts(fact):
    """Extract the components of a PDDL fact by removing parentheses and splitting the string."""
    return fact[1:-1].split()

def match(fact, *args):
    """
    Check if a PDDL fact matches a given pattern.

    - `fact`: The complete fact as a string, e.g., "(at bob shed)".
    - `args`: The expected pattern (wildcards `*` allowed).
    - Returns `True` if the fact matches the pattern, else `False`.
    """
    parts = get_parts(fact)
    return all(fnmatch(part, arg) for part, arg in zip(parts, args))

class SpannerHeuristic(Heuristic):
    """
    A domain-dependent heuristic for the Spanner domain.

    # Summary
    This heuristic estimates the number of actions needed to tighten all loose nuts in the Spanner domain.
    The heuristic considers the following:
    - The man must pick up a usable spanner if not already carrying one.
    - The man must walk to the location of each loose nut.
    - The man must tighten each loose nut using a usable spanner.

    # Assumptions:
    - The man can carry only one spanner at a time.
    - A spanner becomes unusable after tightening a nut.
    - The man must walk to the location of a spanner to pick it up.
    - The man must walk to the location of a nut to tighten it.

    # Heuristic Initialization
    - Extract the goal conditions (tightened nuts) and static facts (links between locations).
    - Build a graph of locations using the static `link` facts to compute distances.

    # Step-By-Step Thinking for Computing Heuristic
    1. Identify the current location of the man.
    2. Identify the locations of all loose nuts.
    3. Identify the locations of all usable spanners.
    4. If the man is not carrying a usable spanner:
       - Compute the distance to the nearest usable spanner.
       - Add the cost of picking up the spanner.
    5. For each loose nut:
       - Compute the distance from the man's current location (or the spanner's location) to the nut's location.
       - Add the cost of walking to the nut's location and tightening it.
    6. Sum the total cost of all actions required to tighten all loose nuts.
    """

    def __init__(self, task):
        """Initialize the heuristic by extracting goal conditions and static facts."""
        self.goals = task.goals  # Goal conditions.
        static_facts = task.static  # Facts that are not affected by actions.

        # Build a graph of locations using the static `link` facts.
        self.location_graph = {}
        for fact in static_facts:
            if match(fact, "link", "*", "*"):
                _, loc1, loc2 = get_parts(fact)
                if loc1 not in self.location_graph:
                    self.location_graph[loc1] = set()
                if loc2 not in self.location_graph:
                    self.location_graph[loc2] = set()
                self.location_graph[loc1].add(loc2)
                self.location_graph[loc2].add(loc1)

    def __call__(self, node):
        """Estimate the minimum cost to tighten all loose nuts."""
        state = node.state

        # Identify the current location of the man.
        man_location = None
        for fact in state:
            if match(fact, "at", "*", "*"):
                _, obj, loc = get_parts(fact)
                if obj == "bob":  # Assuming the man is named "bob".
                    man_location = loc
                    break

        # Identify all loose nuts and their locations.
        loose_nuts = []
        for fact in state:
            if match(fact, "loose", "*"):
                _, nut = get_parts(fact)
                for loc_fact in state:
                    if match(loc_fact, "at", nut, "*"):
                        _, _, nut_loc = get_parts(loc_fact)
                        loose_nuts.append((nut, nut_loc))
                        break

        # Identify all usable spanners and their locations.
        usable_spanners = []
        for fact in state:
            if match(fact, "usable", "*"):
                _, spanner = get_parts(fact)
                for loc_fact in state:
                    if match(loc_fact, "at", spanner, "*"):
                        _, _, spanner_loc = get_parts(loc_fact)
                        usable_spanners.append((spanner, spanner_loc))
                        break

        # Check if the man is carrying a usable spanner.
        carrying_spanner = None
        for fact in state:
            if match(fact, "carrying", "*", "*"):
                _, man, spanner = get_parts(fact)
                if man == "bob":
                    carrying_spanner = spanner
                    break

        # If the man is not carrying a usable spanner, find the nearest one.
        if carrying_spanner is None:
            if not usable_spanners:
                return float('inf')  # No usable spanners available.
            # Compute the distance to the nearest usable spanner.
            nearest_spanner_distance = float('inf')
            for _, spanner_loc in usable_spanners:
                distance = self._compute_distance(man_location, spanner_loc)
                if distance < nearest_spanner_distance:
                    nearest_spanner_distance = distance
            # Add the cost of picking up the spanner.
            total_cost = nearest_spanner_distance + 1
        else:
            total_cost = 0

        # For each loose nut, compute the distance and add the cost of tightening.
        for nut, nut_loc in loose_nuts:
            if carrying_spanner is not None:
                # The man is already carrying a spanner.
                distance = self._compute_distance(man_location, nut_loc)
            else:
                # The man needs to walk from the spanner's location to the nut's location.
                distance = self._compute_distance(spanner_loc, nut_loc)
            total_cost += distance + 1  # Add the cost of tightening the nut.

        return total_cost

    def _compute_distance(self, start, end):
        """
        Compute the shortest path distance between two locations using BFS.

        - `start`: The starting location.
        - `end`: The target location.
        - Returns the number of steps (distance) between the two locations.
        """
        if start == end:
            return 0
        visited = set()
        queue = [(start, 0)]
        while queue:
            current, distance = queue.pop(0)
            if current == end:
                return distance
            if current in visited:
                continue
            visited.add(current)
            for neighbor in self.location_graph.get(current, []):
                queue.append((neighbor, distance + 1))
        return float('inf')  # If no path exists.
