from fnmatch import fnmatch
from heuristics.heuristic_base import Heuristic

def get_parts(fact):
    """Extract the components of a PDDL fact by removing parentheses and splitting the string."""
    return fact[1:-1].split()

def match(fact, *args):
    """
    Check if a PDDL fact matches a given pattern.

    - `fact`: The complete fact as a string, e.g., "(at p1 l1)".
    - `args`: The expected pattern (wildcards `*` allowed).
    - Returns `True` if the fact matches the pattern, else `False`.
    """
    parts = get_parts(fact)
    return all(fnmatch(part, arg) for part, arg in zip(parts, args))

class TransportHeuristic(Heuristic):
    """
    A domain-dependent heuristic for the Transport domain.

    # Summary
    This heuristic estimates the number of actions required to transport all packages to their goal locations. It considers the current state of packages, vehicles, and the road network to compute an efficient estimate.

    # Assumptions
    - Packages can be on the ground or inside vehicles.
    - Vehicles can carry multiple packages, but their capacity is limited.
    - Roads are bidirectional, and vehicles can move freely between connected locations.

    # Heuristic Initialization
    - Extract goal locations for each package from the task's goal conditions.
    - Extract the road network and capacity constraints from the static facts.

    # Step-By-Step Thinking for Computing Heuristic
    1. Identify the current location of each package and its goal location.
    2. Determine if the package is already at its goal location. If so, no action is needed.
    3. If the package is not at its goal:
       - If the package is inside a vehicle, estimate the cost of unloading it.
       - If the package is on the ground, estimate the cost of loading it into a vehicle.
    4. Estimate the cost of moving the vehicle to the goal location, considering the road network.
    5. Sum the costs for all packages to get the total heuristic value.
    """

    def __init__(self, task):
        """
        Initialize the heuristic by extracting:
        - Goal locations for each package.
        - Road network and capacity constraints from static facts.
        """
        self.goals = task.goals  # Goal conditions.
        static_facts = task.static  # Facts that are not affected by actions.

        # Extract road network from static facts.
        self.roads = set()
        for fact in static_facts:
            if match(fact, "road", "*", "*"):
                parts = get_parts(fact)
                self.roads.add((parts[1], parts[2]))

        # Extract capacity constraints from static facts.
        self.capacity_predecessors = {}
        for fact in static_facts:
            if match(fact, "capacity-predecessor", "*", "*"):
                parts = get_parts(fact)
                self.capacity_predecessors[parts[2]] = parts[1]

        # Store goal locations for each package.
        self.goal_locations = {}
        for goal in self.goals:
            predicate, *args = get_parts(goal)
            if predicate == "at":
                package, location = args
                self.goal_locations[package] = location

    def __call__(self, node):
        """Compute an estimate of the minimal number of required actions."""
        state = node.state  # Current world state.

        # Track where packages and vehicles are currently located.
        current_locations = {}
        for fact in state:
            predicate, *args = get_parts(fact)
            if predicate in ["at", "in"]:  # Track both direct location and inside vehicle.
                obj, location = args
                current_locations[obj] = location

        total_cost = 0  # Initialize action cost counter.

        for package, goal_location in self.goal_locations.items():
            # Get the current location of the package (could be a location or a vehicle).
            current_location = current_locations[package]

            # Check if the package is inside a vehicle.
            in_vehicle = current_location.startswith("v")

            if in_vehicle:
                # If the package is inside a vehicle, estimate the cost of unloading it.
                total_cost += 1  # Unload action.

                # Retrieve the physical location of the vehicle.
                current_location = current_locations[current_location]

            # If the package is not at its goal location, estimate the cost of moving it.
            if current_location != goal_location:
                # Estimate the cost of moving the vehicle to the goal location.
                # For simplicity, assume the shortest path between locations is 1 action.
                total_cost += 1  # Drive action.

                # If the package is not in a vehicle, estimate the cost of loading it.
                if not in_vehicle:
                    total_cost += 1  # Pick-up action.

        return total_cost
