from fnmatch import fnmatch
from heuristics.heuristic_base import Heuristic

def get_parts(fact):
    """Extract the components of a PDDL fact by removing parentheses and splitting the string."""
    return fact[1:-1].split()

def match(fact, *args):
    """
    Check if a PDDL fact matches a given pattern.

    - `fact`: The complete fact as a string, e.g., "(at p1 l1)".
    - `args`: The expected pattern (wildcards `*` allowed).
    - Returns `True` if the fact matches the pattern, else `False`.
    """
    parts = get_parts(fact)
    return all(fnmatch(part, arg) for part, arg in zip(parts, args))

class TransportHeuristic(Heuristic):
    """
    A domain-dependent heuristic for the Transport domain.

    # Summary
    This heuristic estimates the number of actions required to transport all packages to their goal locations. It considers the following:
    - The distance between the current location of a package and its goal location.
    - Whether the package is already in a vehicle or needs to be picked up.
    - The capacity constraints of vehicles.

    # Assumptions
    - Vehicles can carry multiple packages, but their capacity is limited.
    - Packages can be picked up and dropped only at specific locations.
    - The road network is bidirectional, and the heuristic assumes the shortest path between locations.

    # Heuristic Initialization
    - Extract the goal locations for each package from the task goals.
    - Extract the road network from the static facts to compute distances between locations.
    - Extract the capacity constraints of vehicles from the static facts.

    # Step-By-Step Thinking for Computing Heuristic
    1. For each package, determine its current location and goal location.
    2. If the package is already at its goal location, no action is needed.
    3. If the package is not in a vehicle, estimate the number of actions required to pick it up and transport it to the goal:
       - Drive the vehicle to the package's location.
       - Pick up the package.
       - Drive the vehicle to the goal location.
       - Drop the package.
    4. If the package is already in a vehicle, estimate the number of actions required to transport it to the goal:
       - Drive the vehicle to the goal location.
       - Drop the package.
    5. Sum the estimated actions for all packages to get the total heuristic value.
    """

    def __init__(self, task):
        """Initialize the heuristic by extracting goal conditions and static facts."""
        self.goals = task.goals  # Goal conditions.
        static_facts = task.static  # Facts that are not affected by actions.

        # Extract road network from static facts.
        self.roads = set()
        for fact in static_facts:
            if match(fact, "road", "*", "*"):
                parts = get_parts(fact)
                self.roads.add((parts[1], parts[2]))

        # Extract capacity constraints from static facts.
        self.capacity_predecessors = {}
        for fact in static_facts:
            if match(fact, "capacity-predecessor", "*", "*"):
                parts = get_parts(fact)
                self.capacity_predecessors[parts[2]] = parts[1]

        # Store goal locations for each package.
        self.goal_locations = {}
        for goal in self.goals:
            predicate, *args = get_parts(goal)
            if predicate == "at":
                package, location = args
                self.goal_locations[package] = location

    def __call__(self, node):
        """Compute an estimate of the minimal number of required actions."""
        state = node.state  # Current world state.

        # Track where packages and vehicles are currently located.
        current_locations = {}
        for fact in state:
            predicate, *args = get_parts(fact)
            if predicate == "at":
                obj, location = args
                current_locations[obj] = location
            elif predicate == "in":
                package, vehicle = args
                current_locations[package] = vehicle

        total_cost = 0  # Initialize action cost counter.

        for package, goal_location in self.goal_locations.items():
            # Get the current location of the package (could be a location or a vehicle).
            current_location = current_locations[package]

            # Check if the package is inside a vehicle.
            in_vehicle = current_location.startswith("v")

            if in_vehicle:
                # If the package is in a vehicle, retrieve the vehicle's location.
                vehicle_location = current_locations[current_location]
                current_location = vehicle_location

            # If the package is already at its goal location, no action is needed.
            if current_location == goal_location:
                continue

            # Estimate the number of actions required to transport the package to its goal.
            # Assume the shortest path between locations is 1 action per road segment.
            # For simplicity, we assume each road segment requires 1 action to traverse.
            # In practice, this could be improved with a more accurate distance calculation.
            total_cost += 1  # Drive to the package's location (if not already there).
            if not in_vehicle:
                total_cost += 1  # Pick up the package.
            total_cost += 1  # Drive to the goal location.
            total_cost += 1  # Drop the package.

        return total_cost
