from fnmatch import fnmatch
from heuristics.heuristic_base import Heuristic

def get_parts(fact):
    """Extract the components of a PDDL fact by removing parentheses and splitting the string."""
    return fact[1:-1].split()

def match(fact, *args):
    """
    Check if a PDDL fact matches a given pattern.

    - `fact`: The complete fact as a string, e.g., "(at p1 l1)".
    - `args`: The expected pattern (wildcards `*` allowed).
    - Returns `True` if the fact matches the pattern, else `False`.
    """
    parts = get_parts(fact)
    return all(fnmatch(part, arg) for part, arg in zip(parts, args))

class TransportHeuristic(Heuristic):
    """
    A domain-dependent heuristic for the Transport domain.

    # Summary
    This heuristic estimates the number of actions required to transport all packages to their goal locations. It considers the current state of packages, vehicles, and the road network to compute an efficient estimate.

    # Assumptions
    - Packages can be on the ground or inside vehicles.
    - Vehicles can carry multiple packages, but their capacity is limited.
    - Roads are bidirectional, and vehicles can move freely between connected locations.

    # Heuristic Initialization
    - Extract goal locations for each package from the task's goal conditions.
    - Extract the road network from static facts to compute distances between locations.
    - Extract vehicle capacities and their current states.

    # Step-By-Step Thinking for Computing Heuristic
    1. For each package, determine its current location and goal location.
    2. If the package is already at its goal, no action is needed.
    3. If the package is inside a vehicle, determine the vehicle's current location.
    4. Compute the shortest path (number of road segments) from the package's current location to its goal location.
    5. If the package is not in a vehicle, estimate the cost of picking it up and transporting it to the goal.
    6. Sum the estimated costs for all packages to get the total heuristic value.
    """

    def __init__(self, task):
        """
        Initialize the heuristic by extracting:
        - Goal locations for each package.
        - Road network from static facts.
        - Vehicle capacities and current states.
        """
        self.goals = task.goals  # Goal conditions.
        static_facts = task.static  # Facts that are not affected by actions.

        # Extract road network from static facts.
        self.road_network = {}
        for fact in static_facts:
            if match(fact, "road", "*", "*"):
                _, l1, l2 = get_parts(fact)
                if l1 not in self.road_network:
                    self.road_network[l1] = set()
                if l2 not in self.road_network:
                    self.road_network[l2] = set()
                self.road_network[l1].add(l2)
                self.road_network[l2].add(l1)

        # Extract goal locations for each package.
        self.goal_locations = {}
        for goal in self.goals:
            if match(goal, "at", "*", "*"):
                _, package, location = get_parts(goal)
                self.goal_locations[package] = location

    def __call__(self, node):
        """Compute an estimate of the minimal number of required actions."""
        state = node.state  # Current world state.

        # Track where packages and vehicles are currently located.
        current_locations = {}
        for fact in state:
            predicate, *args = get_parts(fact)
            if predicate in ["at", "in"]:  # Track both direct location and inside vehicle.
                obj, location = args
                current_locations[obj] = location

        total_cost = 0  # Initialize action cost counter.

        for package, goal_location in self.goal_locations.items():
            # Get the current location of the package (could be a location or inside a vehicle).
            current_location = current_locations[package]

            # Check if the package is inside a vehicle.
            in_vehicle = current_location.startswith("v")

            if in_vehicle:
                # Retrieve the physical location of the vehicle.
                current_location = current_locations[current_location]

            # If the package is already at its goal, no action is needed.
            if current_location == goal_location:
                continue

            # Compute the shortest path from current_location to goal_location.
            # For simplicity, assume each road segment costs 1 action.
            # This can be optimized with a BFS or Dijkstra's algorithm.
            visited = set()
            queue = [(current_location, 0)]
            while queue:
                loc, cost = queue.pop(0)
                if loc == goal_location:
                    total_cost += cost
                    break
                if loc in visited:
                    continue
                visited.add(loc)
                for neighbor in self.road_network.get(loc, []):
                    queue.append((neighbor, cost + 1))

            # If the package is not in a vehicle, add the cost of picking it up.
            if not in_vehicle:
                total_cost += 1  # Cost of pick-up action.

        return total_cost
