from fnmatch import fnmatch
from heuristics.heuristic_base import Heuristic

def get_parts(fact):
    """Extract the components of a PDDL fact by removing parentheses and splitting the string."""
    return fact[1:-1].split()

def match(fact, *args):
    """
    Check if a PDDL fact matches a given pattern.

    - `fact`: The complete fact as a string, e.g., "(on b1 b2)".
    - `args`: The expected pattern (wildcards `*` allowed).
    - Returns `True` if the fact matches the pattern, else `False`.
    """
    parts = get_parts(fact)
    return all(fnmatch(part, arg) for part, arg in zip(parts, args))

class BlocksworldHeuristic(Heuristic):
    """
    A domain-dependent heuristic for the Blocksworld domain.

    # Summary
    This heuristic estimates the number of actions required to reach the goal state by counting the number of blocks that are not in their correct position or not correctly stacked. It also considers the number of blocks that need to be moved to achieve the desired stacking.

    # Assumptions:
    - The goal state specifies the desired stacking of blocks.
    - Blocks can only be moved one at a time.
    - The arm can hold only one block at a time.

    # Heuristic Initialization
    - Extract the goal conditions for each block, specifically the desired "on" relationships.
    - Extract the initial state to determine the current stacking of blocks.

    # Step-By-Step Thinking for Computing Heuristic
    1. Identify the current stacking of blocks and compare it with the goal stacking.
    2. For each block, check if it is in the correct position:
       - If a block is not in the correct position, increment the heuristic value by 1.
    3. For each block that is not directly on the correct block, increment the heuristic value by 1.
    4. If a block is on the table but should be on another block, increment the heuristic value by 1.
    5. If a block is on another block but should be on the table, increment the heuristic value by 1.
    6. The total heuristic value is the sum of all these increments.
    """

    def __init__(self, task):
        """Initialize the heuristic by extracting goal conditions and initial state."""
        self.goals = task.goals  # Goal conditions.
        self.initial_state = task.initial_state  # Initial state.

        # Extract the desired "on" relationships from the goal.
        self.goal_on = {}
        for goal in self.goals:
            predicate, *args = get_parts(goal)
            if predicate == "on":
                block, under_block = args
                self.goal_on[block] = under_block

        # Extract the desired "on-table" conditions from the goal.
        self.goal_on_table = set()
        for goal in self.goals:
            predicate, *args = get_parts(goal)
            if predicate == "on-table":
                block = args[0]
                self.goal_on_table.add(block)

    def __call__(self, node):
        """Estimate the number of actions required to reach the goal state."""
        state = node.state  # Current world state.

        # Track the current "on" relationships.
        current_on = {}
        for fact in state:
            predicate, *args = get_parts(fact)
            if predicate == "on":
                block, under_block = args
                current_on[block] = under_block

        # Track the current "on-table" conditions.
        current_on_table = set()
        for fact in state:
            predicate, *args = get_parts(fact)
            if predicate == "on-table":
                block = args[0]
                current_on_table.add(block)

        # Initialize the heuristic value.
        heuristic_value = 0

        # Check each block's position against the goal.
        for block, goal_under_block in self.goal_on.items():
            if block in current_on:
                if current_on[block] != goal_under_block:
                    heuristic_value += 1
            else:
                heuristic_value += 1

        # Check each block's "on-table" condition against the goal.
        for block in self.goal_on_table:
            if block not in current_on_table:
                heuristic_value += 1

        return heuristic_value
