from fnmatch import fnmatch
from heuristics.heuristic_base import Heuristic

def get_parts(fact):
    """Extract the components of a PDDL fact by removing parentheses and splitting the string."""
    return fact[1:-1].split()

def match(fact, *args):
    """
    Check if a PDDL fact matches a given pattern.

    - `fact`: The complete fact as a string, e.g., "(at_kitchen_bread bread1)".
    - `args`: The expected pattern (wildcards `*` allowed).
    - Returns `True` if the fact matches the pattern, else `False`.
    """
    parts = get_parts(fact)
    return all(fnmatch(part, arg) for part, arg in zip(parts, args))

class ChildsnackHeuristic(Heuristic):
    """
    A domain-dependent heuristic for the Childsnacks domain.

    # Summary
    This heuristic estimates the number of actions required to serve all children with sandwiches, considering the constraints of gluten allergies and the availability of ingredients.

    # Assumptions
    - Each child must be served exactly one sandwich.
    - Children with gluten allergies must be served gluten-free sandwiches.
    - Sandwiches must be made from available bread and content portions in the kitchen.
    - Sandwiches must be placed on trays and moved to the appropriate locations before serving.

    # Heuristic Initialization
    - Extract the goal conditions (all children must be served).
    - Extract static facts (e.g., which children are allergic to gluten, which bread and content portions are gluten-free).
    - Identify the initial state of the kitchen (available bread and content portions, trays, and sandwiches).

    # Step-By-Step Thinking for Computing Heuristic
    1. Count the number of children who are not yet served.
    2. For each unserved child:
       - If the child is allergic to gluten, ensure that a gluten-free sandwich is available or can be made.
       - If the child is not allergic to gluten, any sandwich can be used.
    3. Estimate the number of actions required to:
       - Make sandwiches (either gluten-free or regular).
       - Place sandwiches on trays.
       - Move trays to the appropriate locations.
       - Serve sandwiches to children.
    4. Sum the estimated actions for all unserved children to get the total heuristic value.
    """

    def __init__(self, task):
        """Initialize the heuristic by extracting goal conditions and static facts."""
        self.goals = task.goals  # Goal conditions.
        self.static = task.static  # Static facts.

        # Extract information about children's allergies.
        self.allergic_children = {
            get_parts(fact)[1] for fact in self.static if match(fact, "allergic_gluten", "*")
        }
        self.not_allergic_children = {
            get_parts(fact)[1] for fact in self.static if match(fact, "not_allergic_gluten", "*")
        }

        # Extract information about gluten-free ingredients.
        self.gluten_free_bread = {
            get_parts(fact)[1] for fact in self.static if match(fact, "no_gluten_bread", "*")
        }
        self.gluten_free_content = {
            get_parts(fact)[1] for fact in self.static if match(fact, "no_gluten_content", "*")
        }

    def __call__(self, node):
        """Estimate the number of actions required to serve all children."""
        state = node.state  # Current world state.

        # Count the number of unserved children.
        unserved_children = {
            get_parts(fact)[1] for fact in self.goals if match(fact, "served", "*")
        } - {
            get_parts(fact)[1] for fact in state if match(fact, "served", "*")
        }

        total_cost = 0  # Initialize the heuristic cost.

        for child in unserved_children:
            # Determine if the child is allergic to gluten.
            is_allergic = child in self.allergic_children

            # Estimate the cost to serve this child.
            if is_allergic:
                # Need to make a gluten-free sandwich.
                total_cost += 1  # Make sandwich.
                total_cost += 1  # Put on tray.
                total_cost += 1  # Move tray.
                total_cost += 1  # Serve sandwich.
            else:
                # Can use any sandwich.
                total_cost += 1  # Make sandwich.
                total_cost += 1  # Put on tray.
                total_cost += 1  # Move tray.
                total_cost += 1  # Serve sandwich.

        return total_cost
