from fnmatch import fnmatch
from heuristics.heuristic_base import Heuristic


def get_parts(fact):
    """Extract the components of a PDDL fact by removing parentheses and splitting the string."""
    return fact[1:-1].split()


def match(fact, *args):
    """
    Check if a PDDL fact matches a given pattern.

    - `fact`: The complete fact as a string, e.g., "(robot-at robot1 tile_0_1)".
    - `args`: The expected pattern (wildcards `*` allowed).
    - Returns `True` if the fact matches the pattern, else `False`.
    """
    parts = get_parts(fact)
    return all(fnmatch(part, arg) for part, arg in zip(parts, args))


class FloortileHeuristic(Heuristic):
    """
    A domain-dependent heuristic for the Floortile domain.

    # Summary
    This heuristic estimates the number of actions required to paint all tiles according to the goal configuration.
    It considers the following:
    - The number of tiles that still need to be painted.
    - The distance between the robot's current position and the tiles that need to be painted.
    - The number of color changes required for the robot to paint the tiles with the correct colors.

    # Assumptions
    - The robot can only paint adjacent tiles (up, down, left, right).
    - The robot must change its color if it needs to paint a tile with a different color than the one it currently holds.
    - The robot can move freely between tiles as long as they are clear.

    # Heuristic Initialization
    - Extract the goal conditions for each tile (i.e., the required color for each tile).
    - Extract the static facts (e.g., adjacency relationships between tiles) to compute distances.
    - Initialize data structures to store the goal colors and adjacency relationships.

    # Step-By-Step Thinking for Computing Heuristic
    1. Identify the tiles that still need to be painted (i.e., tiles that are either not painted or painted with the wrong color).
    2. For each such tile, compute the Manhattan distance from the robot's current position to the tile.
    3. If the robot does not have the correct color for a tile, add a cost for changing the color.
    4. Sum the distances and color change costs to estimate the total number of actions required.
    """

    def __init__(self, task):
        """
        Initialize the heuristic by extracting:
        - Goal conditions for each tile.
        - Static facts (adjacency relationships between tiles).
        """
        self.goals = task.goals  # Goal conditions.
        static_facts = task.static  # Facts that are not affected by actions.

        # Extract goal colors for each tile.
        self.goal_colors = {}
        for goal in self.goals:
            predicate, *args = get_parts(goal)
            if predicate == "painted":
                tile, color = args
                self.goal_colors[tile] = color

        # Extract adjacency relationships between tiles.
        self.adjacent = {}
        for fact in static_facts:
            predicate, *args = get_parts(fact)
            if predicate in ["up", "down", "left", "right"]:
                tile1, tile2 = args
                if tile1 not in self.adjacent:
                    self.adjacent[tile1] = []
                self.adjacent[tile1].append(tile2)

    def __call__(self, node):
        """Compute an estimate of the minimal number of required actions."""
        state = node.state  # Current world state.

        # Find the robot's current position and color.
        robot_pos = None
        robot_color = None
        for fact in state:
            predicate, *args = get_parts(fact)
            if predicate == "robot-at":
                robot, pos = args
                robot_pos = pos
            elif predicate == "robot-has":
                robot, color = args
                robot_color = color

        total_cost = 0  # Initialize action cost counter.

        # Iterate over all tiles that need to be painted.
        for tile, goal_color in self.goal_colors.items():
            # Check if the tile is already painted with the correct color.
            painted_correctly = False
            for fact in state:
                predicate, *args = get_parts(fact)
                if predicate == "painted" and args[0] == tile and args[1] == goal_color:
                    painted_correctly = True
                    break

            if not painted_correctly:
                # Compute the Manhattan distance from the robot's current position to the tile.
                # For simplicity, assume tiles are named as "tile_X_Y" where X and Y are coordinates.
                if robot_pos:
                    robot_x, robot_y = map(int, robot_pos.split("_")[1:])
                    tile_x, tile_y = map(int, tile.split("_")[1:])
                    distance = abs(robot_x - tile_x) + abs(robot_y - tile_y)
                else:
                    distance = 0  # If the robot's position is unknown, assume it's already at the tile.

                # Add the distance to the total cost.
                total_cost += distance

                # If the robot does not have the correct color, add a cost for changing the color.
                if robot_color != goal_color:
                    total_cost += 1  # Cost for changing color.

        return total_cost
