from fnmatch import fnmatch
from heuristics.heuristic_base import Heuristic


def get_parts(fact):
    """Extract the components of a PDDL fact by removing parentheses and splitting the string."""
    return fact[1:-1].split()


def match(fact, *args):
    """
    Check if a PDDL fact matches a given pattern.

    - `fact`: The complete fact as a string, e.g., "(at-robot loc_1_1)".
    - `args`: The expected pattern (wildcards `*` allowed).
    - Returns `True` if the fact matches the pattern, else `False`.
    """
    parts = get_parts(fact)
    return all(fnmatch(part, arg) for part, arg in zip(parts, args))


class SokobanHeuristic(Heuristic):
    """
    A domain-dependent heuristic for the Sokoban domain.

    # Summary
    This heuristic estimates the number of actions required to push all boxes to their goal locations.
    It considers the distance between the robot and the boxes, as well as the distance between the boxes and their goal locations.

    # Assumptions
    - The robot can only push one box at a time.
    - The robot must be adjacent to a box to push it.
    - The heuristic does not account for deadlocks or blocked paths.

    # Heuristic Initialization
    - Extract goal locations for each box from the task goals.
    - Extract adjacency relationships between locations from the static facts.

    # Step-By-Step Thinking for Computing Heuristic
    1. For each box, compute the Manhattan distance between its current location and its goal location.
    2. For each box, compute the Manhattan distance between the robot's current location and the box's current location.
    3. Sum these distances to estimate the total number of actions required to move the robot to the box and then push the box to its goal.
    """

    def __init__(self, task):
        """
        Initialize the heuristic by extracting:
        - Goal locations for each box.
        - Adjacency relationships between locations.
        """
        self.goals = task.goals  # Goal conditions.
        static_facts = task.static  # Facts that are not affected by actions.

        # Map locations to their coordinates.
        self.location_to_coords = {}
        for fact in static_facts:
            if match(fact, "adjacent", "*", "*", "*"):
                loc1, loc2, _ = get_parts(fact)[1:]
                if loc1 not in self.location_to_coords:
                    self.location_to_coords[loc1] = self._parse_coords(loc1)
                if loc2 not in self.location_to_coords:
                    self.location_to_coords[loc2] = self._parse_coords(loc2)

        # Store goal locations for each box.
        self.goal_locations = {}
        for goal in self.goals:
            predicate, *args = get_parts(goal)
            if predicate == "at":
                box, location = args
                self.goal_locations[box] = location

    def _parse_coords(self, location):
        """Parse the coordinates from a location name (e.g., 'loc_1_2' -> (1, 2))."""
        parts = location.split("_")
        return (int(parts[1]), int(parts[2]))

    def _manhattan_distance(self, loc1, loc2):
        """Compute the Manhattan distance between two locations."""
        x1, y1 = self.location_to_coords[loc1]
        x2, y2 = self.location_to_coords[loc2]
        return abs(x1 - x2) + abs(y1 - y2)

    def __call__(self, node):
        """Compute an estimate of the minimal number of required actions."""
        state = node.state  # Current world state.

        # Track the current location of the robot and boxes.
        robot_location = None
        box_locations = {}
        for fact in state:
            predicate, *args = get_parts(fact)
            if predicate == "at-robot":
                robot_location = args[0]
            elif predicate == "at":
                box, location = args
                box_locations[box] = location

        total_cost = 0  # Initialize action cost counter.

        for box, goal_location in self.goal_locations.items():
            current_location = box_locations[box]
            if current_location != goal_location:
                # Distance from robot to box.
                robot_to_box = self._manhattan_distance(robot_location, current_location)
                # Distance from box to goal.
                box_to_goal = self._manhattan_distance(current_location, goal_location)
                total_cost += robot_to_box + box_to_goal

        return total_cost
