from fnmatch import fnmatch
from heuristics.heuristic_base import Heuristic

def get_parts(fact):
    """Extract the components of a PDDL fact by removing parentheses and splitting the string."""
    return fact[1:-1].split()

def match(fact, *args):
    """
    Check if a PDDL fact matches a given pattern.

    - `fact`: The complete fact as a string, e.g., "(at p1 l1)".
    - `args`: The expected pattern (wildcards `*` allowed).
    - Returns `True` if the fact matches the pattern, else `False`.
    """
    parts = get_parts(fact)
    return all(fnmatch(part, arg) for part, arg in zip(parts, args))

class TransportHeuristic(Heuristic):
    """
    A domain-dependent heuristic for the Transport domain.

    # Summary
    This heuristic estimates the number of actions required to transport all packages to their goal locations. It considers the current state of packages, vehicles, and roads, and calculates the minimal number of actions needed to achieve the goal.

    # Assumptions
    - Packages can be on the ground or inside vehicles.
    - Vehicles can carry multiple packages, but their capacity is limited.
    - Roads are bidirectional, and vehicles can move freely between connected locations.

    # Heuristic Initialization
    - Extract goal locations for each package from the task goals.
    - Extract static information about roads and capacities from the task's static facts.

    # Step-By-Step Thinking for Computing Heuristic
    1. Identify the current location of each package and its goal location.
    2. Determine if the package is already at its goal location. If so, no action is needed.
    3. If the package is not at its goal location:
       - If the package is inside a vehicle, calculate the cost to unload it.
       - If the package is on the ground, calculate the cost to load it into a vehicle.
    4. Estimate the number of drive actions required to move the vehicle (or package) from its current location to the goal location.
    5. Sum the costs of all necessary actions to get the total heuristic value.
    """

    def __init__(self, task):
        """
        Initialize the heuristic by extracting:
        - Goal locations for each package.
        - Static facts (roads and capacity-predecessor relationships).
        """
        self.goals = task.goals  # Goal conditions.
        static_facts = task.static  # Facts that are not affected by actions.

        # Extract roads from static facts.
        self.roads = set()
        for fact in static_facts:
            if match(fact, "road", "*", "*"):
                parts = get_parts(fact)
                self.roads.add((parts[1], parts[2]))

        # Extract capacity-predecessor relationships.
        self.capacity_predecessors = {}
        for fact in static_facts:
            if match(fact, "capacity-predecessor", "*", "*"):
                parts = get_parts(fact)
                self.capacity_predecessors[parts[1]] = parts[2]

        # Store goal locations for each package.
        self.goal_locations = {}
        for goal in self.goals:
            predicate, *args = get_parts(goal)
            if predicate == "at":
                package, location = args
                self.goal_locations[package] = location

    def __call__(self, node):
        """Compute an estimate of the minimal number of required actions."""
        state = node.state  # Current world state.

        # Track where packages and vehicles are currently located.
        current_locations = {}
        for fact in state:
            predicate, *args = get_parts(fact)
            if predicate == "at":
                obj, location = args
                current_locations[obj] = location
            elif predicate == "in":
                package, vehicle = args
                current_locations[package] = vehicle

        total_cost = 0  # Initialize action cost counter.

        for package, goal_location in self.goal_locations.items():
            # Get the current location of the package (could be a location or a vehicle).
            current_location = current_locations.get(package, None)

            if current_location is None:
                continue  # Package not found in state (should not happen).

            # Check if the package is already at its goal location.
            if current_location == goal_location:
                continue  # No action needed.

            # Check if the package is inside a vehicle.
            in_vehicle = current_location.startswith("v")

            if in_vehicle:
                # Package is inside a vehicle; unload it first.
                total_cost += 1  # Cost to unload the package.
                current_location = current_locations[current_location]  # Get vehicle's location.

            # Estimate the number of drive actions to move from current_location to goal_location.
            # This is a simple heuristic: assume the minimal number of roads to traverse.
            # In practice, this could be improved with a shortest-path algorithm.
            if current_location != goal_location:
                total_cost += 1  # Assume at least one drive action.

        return total_cost
