from fnmatch import fnmatch
from heuristics.heuristic_base import Heuristic

def get_parts(fact):
    """Extract the components of a PDDL fact by removing parentheses and splitting the string."""
    return fact[1:-1].split()

def match(fact, *args):
    """
    Check if a PDDL fact matches a given pattern.

    - `fact`: The complete fact as a string, e.g., "(at p1 l1)".
    - `args`: The expected pattern (wildcards `*` allowed).
    - Returns `True` if the fact matches the pattern, else `False`.
    """
    parts = get_parts(fact)
    return all(fnmatch(part, arg) for part, arg in zip(parts, args))

class TransportHeuristic(Heuristic):
    """
    A domain-dependent heuristic for the Transport domain.

    # Summary
    This heuristic estimates the number of actions required to transport all packages to their goal locations. It considers the current state of packages, vehicles, and the road network to compute an efficient estimate.

    # Assumptions
    - Packages can be either on the ground or inside a vehicle.
    - Vehicles can carry multiple packages, but their capacity is limited.
    - The road network is bidirectional, and vehicles can move freely between connected locations.

    # Heuristic Initialization
    - Extract goal locations for each package from the task's goals.
    - Extract the road network and capacity information from the static facts.

    # Step-By-Step Thinking for Computing Heuristic
    1. For each package, determine its current location and goal location.
    2. If the package is already at its goal location, no action is needed.
    3. If the package is inside a vehicle, determine the vehicle's current location.
    4. Compute the shortest path (number of road segments) from the package's current location to its goal location.
    5. If the package is not in a vehicle, estimate the cost of picking it up and transporting it to the goal.
    6. Sum the estimated costs for all packages to get the total heuristic value.
    """

    def __init__(self, task):
        """
        Initialize the heuristic by extracting:
        - Goal locations for each package.
        - Road network and capacity information from static facts.
        """
        self.goals = task.goals  # Goal conditions.
        static_facts = task.static  # Facts that are not affected by actions.

        # Extract road network from static facts.
        self.road_network = {}
        for fact in static_facts:
            if match(fact, "road", "*", "*"):
                _, l1, l2 = get_parts(fact)
                if l1 not in self.road_network:
                    self.road_network[l1] = set()
                if l2 not in self.road_network:
                    self.road_network[l2] = set()
                self.road_network[l1].add(l2)
                self.road_network[l2].add(l1)

        # Extract goal locations for each package.
        self.goal_locations = {}
        for goal in self.goals:
            if match(goal, "at", "*", "*"):
                _, package, location = get_parts(goal)
                self.goal_locations[package] = location

    def __call__(self, node):
        """Compute an estimate of the minimal number of required actions."""
        state = node.state  # Current world state.

        # Track where packages and vehicles are currently located.
        current_locations = {}
        for fact in state:
            predicate, *args = get_parts(fact)
            if predicate in ["at", "in"]:  # Track both direct location and inside vehicle.
                obj, location = args
                current_locations[obj] = location

        total_cost = 0  # Initialize action cost counter.

        for package, goal_location in self.goal_locations.items():
            # Get the current location of the package (could be a location or a vehicle).
            current_location = current_locations[package]

            # Check if the package is inside a vehicle.
            in_vehicle = current_location.startswith("v")

            if in_vehicle:
                # Retrieve the physical location of the vehicle.
                current_location = current_locations[current_location]

            # If the package is already at its goal, no action is needed.
            if current_location == goal_location:
                continue

            # Compute the shortest path from current location to goal location.
            path_length = self._shortest_path_length(current_location, goal_location)

            # Estimate the cost of transporting the package.
            if in_vehicle:
                # If the package is already in a vehicle, only driving actions are needed.
                total_cost += path_length
            else:
                # If the package is not in a vehicle, add the cost of picking it up.
                total_cost += 1 + path_length

        return total_cost

    def _shortest_path_length(self, start, goal):
        """
        Compute the shortest path length between two locations using BFS.

        @param start: The starting location.
        @param goal: The goal location.
        @return: The number of road segments in the shortest path.
        """
        if start == goal:
            return 0

        visited = set()
        queue = [(start, 0)]

        while queue:
            current, distance = queue.pop(0)
            if current == goal:
                return distance
            if current in visited:
                continue
            visited.add(current)
            for neighbor in self.road_network.get(current, []):
                queue.append((neighbor, distance + 1))

        return float('inf')  # If no path exists, return infinity.
