from fnmatch import fnmatch
from heuristics.heuristic_base import Heuristic

def get_parts(fact):
    """Extract the components of a PDDL fact by removing parentheses and splitting the string."""
    return fact[1:-1].split()

def match(fact, *args):
    """
    Check if a PDDL fact matches a given pattern.

    - `fact`: The complete fact as a string, e.g., "(at p1 l1)".
    - `args`: The expected pattern (wildcards `*` allowed).
    - Returns `True` if the fact matches the pattern, else `False`.
    """
    parts = get_parts(fact)
    return all(fnmatch(part, arg) for part, arg in zip(parts, args))

class TransportHeuristic(Heuristic):
    """
    A domain-dependent heuristic for the Transport domain.

    # Summary
    This heuristic estimates the number of actions required to transport all packages to their goal locations. It considers the current state of packages (whether they are in a vehicle or at a location) and the road network to compute the minimal number of actions.

    # Assumptions
    - Packages can be at a location or inside a vehicle.
    - Vehicles can move between connected locations via roads.
    - The capacity of vehicles is respected when picking up or dropping packages.
    - The heuristic does not need to be admissible, so it can overestimate the number of actions.

    # Heuristic Initialization
    - Extract the goal locations for each package from the task goals.
    - Extract the road network from the static facts to determine connectivity between locations.
    - Extract the capacity relationships between sizes from the static facts.

    # Step-By-Step Thinking for Computing Heuristic
    1. For each package, determine its current location (either directly or via the vehicle it is in).
    2. If the package is already at its goal location, no actions are needed.
    3. If the package is in a vehicle, determine the vehicle's current location.
    4. Compute the minimal number of drive actions required to move the vehicle (or package) to the goal location, considering the road network.
    5. If the package is not in a vehicle, compute the minimal number of pick-up and drop actions required to transport it to the goal location.
    6. Sum the actions for all packages to get the total heuristic value.
    """

    def __init__(self, task):
        """
        Initialize the heuristic by extracting:
        - Goal locations for each package.
        - Road network from static facts.
        - Capacity relationships from static facts.
        """
        self.goals = task.goals  # Goal conditions.
        static_facts = task.static  # Facts that are not affected by actions.

        # Extract road network from static facts.
        self.roads = set()
        for fact in static_facts:
            if match(fact, "road", "*", "*"):
                parts = get_parts(fact)
                self.roads.add((parts[1], parts[2]))

        # Extract capacity relationships from static facts.
        self.capacity_predecessors = {}
        for fact in static_facts:
            if match(fact, "capacity-predecessor", "*", "*"):
                parts = get_parts(fact)
                self.capacity_predecessors[parts[2]] = parts[1]

        # Store goal locations for each package.
        self.goal_locations = {}
        for goal in self.goals:
            predicate, *args = get_parts(goal)
            if predicate == "at":
                package, location = args
                self.goal_locations[package] = location

    def __call__(self, node):
        """Compute an estimate of the minimal number of required actions."""
        state = node.state  # Current world state.

        # Track where packages and vehicles are currently located.
        current_locations = {}
        for fact in state:
            predicate, *args = get_parts(fact)
            if predicate in ["at", "in"]:  # Track both direct location and inside vehicle.
                obj, location = args
                current_locations[obj] = location

        total_cost = 0  # Initialize action cost counter.

        for package, goal_location in self.goal_locations.items():
            # Get the current location of the package (could be a location or a vehicle).
            current_location = current_locations[package]

            # Check if the package is inside a vehicle.
            in_vehicle = current_location.startswith("v")

            if in_vehicle:
                # Retrieve the physical location of the vehicle.
                current_location = current_locations[current_location]

            # If the package is already at its goal, no actions are needed.
            if current_location == goal_location:
                continue

            # Compute the minimal number of drive actions to reach the goal location.
            # This is a simplified estimate assuming a direct path exists.
            # In practice, a more accurate pathfinding algorithm could be used.
            drive_cost = 1  # Assume at least one drive action is needed.

            # If the package is not in a vehicle, add pick-up and drop actions.
            if not in_vehicle:
                total_cost += 1  # Pick-up action.
                total_cost += 1  # Drop action.

            total_cost += drive_cost

        return total_cost
