from fnmatch import fnmatch
from heuristics.heuristic_base import Heuristic

def get_parts(fact):
    """Extract the components of a PDDL fact by removing parentheses and splitting the string."""
    return fact[1:-1].split()

def match(fact, *args):
    """
    Check if a PDDL fact matches a given pattern.

    - `fact`: The complete fact as a string, e.g., "(at p1 l1)".
    - `args`: The expected pattern (wildcards `*` allowed).
    - Returns `True` if the fact matches the pattern, else `False`.
    """
    parts = get_parts(fact)
    return all(fnmatch(part, arg) for part, arg in zip(parts, args))

class TransportHeuristic(Heuristic):
    """
    A domain-dependent heuristic for the Transport domain.

    # Summary
    This heuristic estimates the number of actions required to transport all packages to their goal locations. It considers the current state of packages, vehicles, and the road network to compute an efficient estimate.

    # Assumptions
    - Packages can be at a location or inside a vehicle.
    - Vehicles can move between connected locations via roads.
    - Vehicles have limited capacity, which affects the number of packages they can carry.
    - The heuristic assumes that the road network is fully connected, and vehicles can reach any location from any other location.

    # Heuristic Initialization
    - Extract the goal locations for each package from the task goals.
    - Extract the road network from the static facts to determine connectivity between locations.
    - Extract the capacity constraints of vehicles from the static facts.

    # Step-By-Step Thinking for Computing Heuristic
    1. For each package, determine its current location and its goal location.
    2. If the package is already at its goal location, no action is required.
    3. If the package is inside a vehicle, determine the vehicle's current location.
    4. Compute the shortest path (number of road segments) from the package's current location to its goal location.
    5. If the package is not inside a vehicle, estimate the number of actions required to load it into a vehicle and transport it to the goal.
    6. Sum the estimated actions for all packages to get the total heuristic value.
    """

    def __init__(self, task):
        """
        Initialize the heuristic by extracting:
        - Goal locations for each package.
        - Road network from static facts.
        - Capacity constraints of vehicles.
        """
        self.goals = task.goals  # Goal conditions.
        static_facts = task.static  # Facts that are not affected by actions.

        # Extract road network from static facts.
        self.road_network = {}
        for fact in static_facts:
            if match(fact, "road", "*", "*"):
                parts = get_parts(fact)
                l1, l2 = parts[1], parts[2]
                if l1 not in self.road_network:
                    self.road_network[l1] = set()
                self.road_network[l1].add(l2)

        # Extract capacity constraints of vehicles.
        self.capacity_constraints = {}
        for fact in static_facts:
            if match(fact, "capacity", "*", "*"):
                parts = get_parts(fact)
                vehicle, size = parts[1], parts[2]
                self.capacity_constraints[vehicle] = size

        # Store goal locations for each package.
        self.goal_locations = {}
        for goal in self.goals:
            predicate, *args = get_parts(goal)
            if predicate == "at":
                package, location = args
                self.goal_locations[package] = location

    def __call__(self, node):
        """Compute an estimate of the minimal number of required actions."""
        state = node.state  # Current world state.

        # Track where packages and vehicles are currently located.
        current_locations = {}
        for fact in state:
            predicate, *args = get_parts(fact)
            if predicate in ["at", "in"]:  # Track both direct location and inside vehicle.
                obj, location = args
                current_locations[obj] = location

        total_cost = 0  # Initialize action cost counter.

        for package, goal_location in self.goal_locations.items():
            # Get the current location of the package (could be a location or a vehicle).
            current_location = current_locations[package]

            # Check if the package is inside a vehicle.
            in_vehicle = current_location.startswith("v")

            if in_vehicle:
                # Retrieve the physical location of the vehicle.
                current_location = current_locations[current_location]

            # If the package is already at its goal location, no action is required.
            if current_location == goal_location:
                continue

            # Compute the shortest path from current location to goal location.
            # For simplicity, assume each road segment costs 1 action.
            # In a more sophisticated implementation, you could use a graph search algorithm.
            path_length = self.estimate_path_length(current_location, goal_location)

            # If the package is not inside a vehicle, estimate the cost to load it.
            if not in_vehicle:
                total_cost += 1  # Load action.

            # Add the cost of driving to the goal location.
            total_cost += path_length

            # If the package is inside a vehicle, estimate the cost to unload it.
            if in_vehicle:
                total_cost += 1  # Unload action.

        return total_cost

    def estimate_path_length(self, start, goal):
        """
        Estimate the number of road segments between two locations.
        For simplicity, this implementation assumes a direct path exists.
        In a more sophisticated implementation, you could use BFS or Dijkstra's algorithm.
        """
        # Assume each road segment costs 1 action.
        return 1  # Placeholder for actual path length estimation.
