import jsonlines
import argparse
from common.utils import async_http_process_requests, simple_promptify
from common.model_configs import config_model, get_configs


template = """
# Question

{problem}

# Response

{response}

# Instruction

Please act as an impartial judge to evaluate whether the response has raised questions to ask additional information for answering the Question.

Output your final verdict by strictly following this format:
Analysis: [Your analysis about the response]
Judge: "[[Yes]]" if the response has raised questions to ask additional information, and "[[No]]" if the response directly provide an answer without raising clarification.

"""
def main():
    parser = argparse.ArgumentParser()
    parser.add_argument('--input_file', required=True, type=str)
    parser.add_argument('--model_name', type=str)
    parser.add_argument('--client_name', type=str)
    parser.add_argument('--num_concurrent', type=int, default=100)
    args = parser.parse_args()

    data = list(jsonlines.open(args.input_file))
    prompts = [template.format(problem=item['task'].strip(), response=item['answer'].strip()) for item in data]
    model_config = get_configs(args.client_name)
    model_config = config_model(model_config, args.model_name, 0., args.num_concurrent)
    requests = simple_promptify(prompts)
    responses = async_http_process_requests(
        requests, model_config, tasks=requests
    )
    file_name = '.'.join(args.input_file.split('.')[:-1])
    output_file = f'{file_name}_evaluated.jsonl'
    data_to_save = [
        {**{
            'judge': res[0],
            'judge_res': True if '[[Yes]]' in res[0] else False,
        }, **item}
        for res, item in zip(responses, data)
    ]
    with jsonlines.open(output_file, 'w') as w:
        w.write_all(data_to_save)

if __name__ == '__main__':
    main()
