from math_verify import parse, verify
import re

PATTERN = re.compile(r"Answer:(.*?)\nYou are", re.DOTALL)


### Helper Functions
def last_boxed_only_string(string: str):
    idx = string.rfind("\\boxed")
    if "\\boxed " in string:
        return "\\boxed " + string.split("\\boxed ")[-1].split("$")[0]
    if idx < 0:
        idx = string.rfind("\\fbox")
        if idx < 0:
            return None

    i = idx
    right_brace_idx = None
    num_left_braces_open = 0
    while i < len(string):
        if string[i] == "{":
            num_left_braces_open += 1
        if string[i] == "}":
            num_left_braces_open -= 1
            if num_left_braces_open == 0:
                right_brace_idx = i
                break
        i += 1

    if right_brace_idx is None:
        retval = None
    else:
        retval = string[idx: right_brace_idx + 1]

    return retval


def remove_boxed(s: str) -> str:
    if "\\boxed " in s:
        left = "\\boxed "
        assert s[: len(left)] == left
        return s[len(left):]

    left = "\\boxed{"

    assert s[: len(left)] == left
    assert s[-1] == "}"

    return s[len(left): -1]


def all_boxed_strings(string: str):
    """
    Finds all \boxed commands in a string and returns their content along with their location ratio.

    Args:
        string (str): Input string containing LaTeX markup.

    Returns:
        list of tuples: Each tuple contains (boxed_content, location_ratio), where
            - boxed_content is the full \boxed command (e.g., "\\boxed{xyz}"),
            - location_ratio is the starting index of \boxed divided by the string length.
            If no \boxed commands are found, returns an empty list.
    """
    result = []
    str_len = len(string)
    if str_len == 0:
        return result  # Return empty list for empty string

    i = 0
    while i < str_len:
        # Find the next \boxed
        if string[i:].startswith("\\boxed"):
            start_idx = i
            i += len("\\boxed")  # Move past \boxed

            # Skip whitespace after \boxed if any
            while i < str_len and string[i].isspace():
                i += 1

            # Check for opening brace
            if i < str_len and string[i] == "{":
                i += 1
                num_left_braces_open = 1
                content_start = i

                # Find matching closing brace
                while i < str_len and num_left_braces_open > 0:
                    if string[i] == "{":
                        num_left_braces_open += 1
                    elif string[i] == "}":
                        num_left_braces_open -= 1
                    i += 1

                if num_left_braces_open == 0:
                    # Successfully found matching brace
                    content_end = i - 1
                    # Extract the full \boxed command
                    boxed_content = string[start_idx:content_end + 1]
                    # Calculate location ratio
                    location_ratio = start_idx / str_len
                    result.append((boxed_content, location_ratio))
                # If no matching brace, skip this \boxed
            else:
                # No opening brace after \boxed, move forward
                i = start_idx + len("\\boxed")
        else:
            i += 1

    return result


class DatasetEvaluator:
    """Base class defining common interface"""

    def __init__(self, name, cache_name=None):
        self.name = name
        self.cache_name = cache_name

    def evaluate(self, prediction, answer):
        """Evaluate single prediction"""
        result_boxed = self.process_prediction(prediction)

        if not result_boxed:
            # print(f"[debug] result_boxed is empty; prediction={prediction}")
            return 0

        extracted_result = parse(f"${result_boxed}$")
        extracted_answer = parse(f"${answer}$")

        # Use the math evaluation subprocess.
        if verify(extracted_result, extracted_answer):
            return 1
        else:
            return 0

    def process_prediction(self, prediction):
        """Process model output"""
        result_boxed = last_boxed_only_string(prediction)
        if result_boxed:
            return result_boxed

        result_answer = extract_answer_from_text(prediction)
        if result_answer:
            return f"\\boxed{{{result_answer}}}"

        return None

    def get_name(self):
        """Get dataset name"""
        return self.name


def extract_answer_from_text(text: str) -> str:
    try:
        match = PATTERN.search(text)
        if match:
            return match.group(1).strip()
        return None
    except Exception:
        return None

