import random

import jsonlines
import argparse
from common.utils import async_http_process_requests, simple_promptify
from common.model_configs import config_model, get_configs


template = """
# Problem to Solve

{problem}

# Instruction

If you need ask information, please raise clarification question and start your response STRICTLY with: "Clarification Question" followed by your questions.
Otherwise, please reason step by step, and put your final answer within \\boxed{{}}.
"""


# basic_template = """
# # Problem to Solve
#
# {problem}
#
# # Instruction
#
# Otherwise, please reason step by step, and put your final answer within \\boxed{{}}.
#
# """

basic_template = """
{problem}
"""

def main():
    parser = argparse.ArgumentParser()
    parser.add_argument('--input_file', type=str)
    parser.add_argument('--use_raw', action='store_true')
    parser.add_argument('--save_path', type=str)
    parser.add_argument('--model_name', type=str)
    parser.add_argument('--client_name', type=str)
    parser.add_argument('--num_concurrent', type=int, default=100)
    parser.add_argument('--basic_prompt', action='store_true')
    parser.add_argument('--max_queries', type=int, default=-1)
    parser.add_argument('--temperature', type=float, default=0)
    args = parser.parse_args()
    with jsonlines.open(args.input_file, 'r') as reader:
        data = list(reader)
    if args.max_queries != -1:
        random.seed(42)
        random.shuffle(data)
        data = data[:args.max_queries]
    if args.use_raw:
        tasks = [template.format(problem=item['raw_task']) if not args.basic_prompt else basic_template.format(problem=item['raw_task']) for item in data]
    else:
        tasks = [template.format(problem=item['reassemble_task']) if not args.basic_prompt else basic_template.format(problem=item['reassemble_task']) for item in data]
    model_config = get_configs(args.client_name)
    model_config = config_model(model_config, args.model_name, args.temperature, args.num_concurrent)
    requests = simple_promptify(tasks)
    responses = async_http_process_requests(
        requests, model_config, tasks=tasks, metadatas=data, save_file=args.save_path
    )
    data_to_save = [
        {
            'task': task,
            'answer': response[0],
            'thought': response[1],
            'usage': response[2],
            'metadata': item
        } if response is not None else {
            'task': task,
            'answer': None,
            'thought': None,
            'usage': None,
            'metadata': item
        }
        for task, response, item in zip(tasks, responses, data)
    ]
    with jsonlines.open(args.save_path, 'w') as writer:
        writer.write_all(data_to_save)


if __name__ == "__main__":
    main()
