import jsonlines
import argparse
import random
from common.utils import async_http_process_requests, simple_promptify, list_to_string
from common.model_configs import config_model, config_aliyun


template = """
# Information

{information}

# Background

{background}

# Instruction

Assemble the Information and Background into a coherent text.

Generate your output STRICTLY in the following format.
Assembled Text:

# Requirements

1. **The Assembled Text SHOULD NOT CONTAIN the title of Information and Background.**
1. **The Assembled Text SHOULD CONTAIN ALL contents in the Information and Background (if the Background is not None).**
2. **DO NOT ADD ANY other content that is not included in the Information and Background to the Assembled Text.**
"""


def parse_results(result, metadata):
    return {
        'response': result,
        'metadata': metadata
    }


def main():
    random.seed(42)
    parser = argparse.ArgumentParser()
    parser.add_argument('--input_file', type=str, required=True)
    parser.add_argument('--output_file', type=str, required=True)
    parser.add_argument('--max_queries', type=int, default=-1)
    parser.add_argument('--seed', type=int, default=42)
    args = parser.parse_args()
    random.seed(args.seed)
    with jsonlines.open(args.input_file) as r:
        data = list(r)
    all_queries, all_metadata = [], []
    if args.max_queries != -1:
        random.shuffle(data)
        data = data[args.max_queries*2:args.max_queries*5]
    for item in data:
        goal, information, background = item['goal'], item['filtered_info_list'], item['background']
        query = template.format(
            information=list_to_string(information, '-'), background=background,
        )
        all_queries.append(query)
        all_metadata.append(item)
        if len(all_queries) == 1:
            print(query)
    all_requests = simple_promptify(all_queries)
    model_config = config_model(config_aliyun, 'deepseek-r1', 0.6, 100)
    all_res = async_http_process_requests(all_requests, model_config)
    all_res = [parse_results(res[0], metadata) for res, metadata in zip(all_res, all_metadata)]
    with jsonlines.open(args.output_file, 'w') as writer:
        writer.write_all(all_res)



if __name__ == '__main__':
    main()
