"""
"""

import pickle
import numpy as np
import torch
import json
from torch.utils.data import Dataset, DataLoader
from BayesianDTI.datahelper import *
from blitz.modules import BayesianLinear, BayesianConv1d, BayesianEmbedding
from blitz.utils import variational_estimator
from abc import ABC, abstractmethod
import torch
import torch.distributions.studentT as studentT
import torch.nn.functional as F
import torch.nn as nn
##################################################################################################################
#
# Abstract classes for torch models.
#
##################################################################################################################

class AbstractMoleculeEncoder(ABC, torch.nn.Module):
    """
    Abstract base class of molecule embedding models.
    """

    def forward(self, src):
        emb = None
        return emb


class AbstractProteinEncoder(ABC, torch.nn.Module):
    """
    Abstract base class of protein embedding models.
    """

    def forward(self, src):
        emb = None
        return emb


class AbstractInteractionModel(ABC, torch.nn.Module):
    """
    Abstract base class of drug-target interaction models.
    """
    def forward(self, protein_emb, drug_emb):
        prediction = None
        return prediction


class AbstractDTIModel(ABC, torch.nn.Module):
    def __init__(self):
        super(AbstractDTIModel, self).__init__()
        self.protein_encoder = AbstractMoleculeEncoder()
        self.smiles_encoder = AbstractProteinEncoder()
        self.interaction_predictor = AbstractInteractionModel()

    def forward(self, d, p):
        """
        Args:
            d(Tensor) : Preprocessed drug input batch
            p(Tensor) : Preprocessed protein input batch

            both d and p contains Long elements representing the token,
            such as
            ["C", "C", "O", "H"] -> Tensor([4, 4, 5, 7])
            ["P, K"] -> Tensor([12, 8])

        Return:
            (Tensor) [batch_size, 1]: predicted affinity value
        """
        p_emb = self.protein_encoder(p)
        d_emb = self.smiles_encoder(d)

        return self.interaction_predictor(p_emb, d_emb)

##################################################################################################
##################################################################################################
##################################################################################################

class MLPMixedDTI(torch.nn.Module):
    def __init__(self, token_len = 64+25+3):
        super(MLPMixedDTI, self).__init__()
        self.token_len = token_len
        self.seqlen = 1286
        self.channel_dim = 128
        self.hidden_dim = 512
        self.embedding = torch.nn.Embedding(token_len, self.channel_dim) #* VOCALEN: chem(64), prot(25), unk(1), sep(1), pedding(1)
        self.fc1 = torch.nn.Linear(self.seqlen, self.hidden_dim) #* VOCALEN: chem, prot, unk, sep

    def forward(self, d, p):
        sep_token = ( torch.ones((d.size(0), 1)).to(d.device)*(self.token_len) ).long()
        print(sep_token[0])
        cat = torch.cat((d, sep_token, p), dim=1)
        print(cat[0])
        out = self.embedding(cat)
        print(out.shape)
        out = out.transpose(1,2)
        print(out.shape)
        out = self.fc1(out)
        print(out.shape)


class SMILESEncoder(AbstractMoleculeEncoder):
    def __init__(self, smile_len=64+1, latent_len=128): ## +1 for 0 padding
        super(SMILESEncoder, self).__init__()
        self.activation_ort = None
        self.encoder = torch.nn.Embedding(smile_len, latent_len)
        self.conv1 = torch.nn.Conv1d(latent_len, 32, 4)
        self.conv2 = torch.nn.Conv1d(32, 64, 6)
        self.conv3 = torch.nn.Conv1d(64, 96, 8)

    def forward(self, src, delta=None):
        emb = self.encoder(src)
        conv1 = torch.nn.ReLU()(self.conv1(emb.transpose(1,2)))
        conv2 = torch.nn.ReLU()(self.conv2(conv1))
        self.activation_ort = activation_similarity(conv2)
        if not (delta is None):
            conv2 = conv2 * delta
        conv3 = torch.nn.ReLU()(self.conv3(conv2))

        return torch.max(conv3, 2)[0]


@variational_estimator
class BayesianSMILESEncoder(SMILESEncoder):
    def __init__(self, smile_len=64+1, latent_len=128): ## +1 for 0 padding
        super(BayesianSMILESEncoder, self).__init__()
        self.encoder = BayesianEmbedding(smile_len, latent_len)
        self.conv1 = BayesianConv1d(latent_len, 32, 4)
        self.conv2 = BayesianConv1d(32, 64, 6)
        self.conv3 = BayesianConv1d(64, 96, 8)


class ProteinEncoder(AbstractProteinEncoder):
    def __init__(self, protein_len=25+1, latent_len=128): ## +1 for 0 padding
        super(ProteinEncoder, self).__init__()
        self.activation_ort = None
        self.encoder = torch.nn.Embedding(protein_len, latent_len)
        self.conv1 = torch.nn.Conv1d(latent_len, 32, 4)
        self.conv2 = torch.nn.Conv1d(32, 64, 8)
        self.conv3 = torch.nn.Conv1d(64, 96, 12)

    def forward(self, src, delta=None):
        emb = self.encoder(src)
        conv1 = torch.nn.ReLU()(self.conv1(emb.transpose(1,2)))
        conv2 = torch.nn.ReLU()(self.conv2(conv1))
        self.activation_ort = activation_similarity(conv2)
        if not (delta is None):
            conv2 = conv2 * delta
        conv3 = torch.nn.ReLU()(self.conv3(conv2))

        return torch.max(conv3, 2)[0]


@variational_estimator
class BayesianProteinEncoder(ProteinEncoder):
    def __init__(self, protein_len=25+1, latent_len=128): ## +1 for 0 padding
        super(BayesianProteinEncoder, self).__init__()
        self.encoder = BayesianEmbedding(protein_len, latent_len)
        self.conv1 = BayesianConv1d(latent_len, 32, 4)
        self.conv2 = BayesianConv1d(32, 64, 8)
        self.conv3 = BayesianConv1d(64, 96, 12)


class InteractionPredictor(AbstractInteractionModel):
    def __init__(self, input_dim):
        super(InteractionPredictor, self).__init__()
        self.fully1 = torch.nn.Linear(input_dim, 1024)
        self.fully2 = torch.nn.Linear(1024, 1024)
        self.fully3 = torch.nn.Linear(1024, 512)
        self.output = torch.nn.Linear(512, 1)
        self.dropout = torch.nn.Dropout(0.1)


    def forward(self, protein_emb, drug_emb, delta=None):
        src = torch.cat((protein_emb, drug_emb), 1)
        #if not (delta is None):
        #    src = src * delta
        fully1 = torch.nn.ReLU()(self.fully1(src))
        fully1 = self.dropout(fully1)
        fully2 = torch.nn.ReLU()(self.fully2(fully1))
        fully2 = self.dropout(fully2)
        fully3 = torch.nn.ReLU()(self.fully3(fully2))
        return self.output(fully3)


class DeepDTA(AbstractDTIModel):
    """
    The final DeepDTA model includes the protein encoding model;
    the smiles(drug; chemical) encoding model; the interaction model.
    """
    def __init__(self, concat_dim=96*2):
        super(DeepDTA, self).__init__()
        self.protein_encoder = ProteinEncoder()
        self.smiles_encoder = SMILESEncoder()
        self.interaction_predictor = InteractionPredictor(concat_dim)
        self.activation_ort = None
    def forward(self, d, p, delta = None):
        """
        Args:
            d(Tensor) : Preprocessed drug input batch
            p(Tensor) : Preprocessed protein input batch

            both d and p contains Long elements representing the token,
            such as
            ["C", "C", "O", "H"] -> Tensor([4, 4, 5, 7])
            ["P, K"] -> Tensor([12, 8])

        Return:
            (Tensor) [batch_size, 1]: predicted affinity value
        """
        p_emb = self.protein_encoder(p, delta=delta)
        d_emb = self.smiles_encoder(d, delta=delta)

        if self.training:
            self.activation_ort = self.protein_encoder.activation_ort
            self.activation_ort += self.smiles_encoder.activation_ort

        return self.interaction_predictor(p_emb, d_emb)

    def train_dropout(self):
        def turn_on_dropout(m):
            if type(m) == torch.nn.modules.dropout.Dropout:
                m.train()
        self.apply(turn_on_dropout)


@variational_estimator
class InteractionEpistemicUncertaintyPredictor(AbstractInteractionModel):
    """
    Interaction layers to estimate Epistemic uncertainty using Bayesian fully conneceted layers
    using Bayes By Backprop(with blitz library).

    """
    def __init__(self, input_dim):
        super(InteractionEpistemicUncertaintyPredictor, self).__init__()
        self.fully1 = torch.nn.Linear(input_dim, 1024)#, prior_sigma_1=0.1, prior_sigma_2 = 0.0002)
        self.fully2 = torch.nn.Linear(1024, 1024)#, prior_sigma_1 = 0.1, prior_sigma_2 = 0.0002)
        self.fully3 = BayesianLinear(1024, 512)
        self.output = BayesianLinear(512, 1)


    def forward(self, protein_emb, drug_emb):
        src = torch.cat((protein_emb, drug_emb), 1)
        fully1 = torch.nn.ReLU()(self.fully1(src))
        fully2 = torch.nn.ReLU()(self.fully2(fully1))
        fully3 = torch.nn.ReLU()(self.fully3(fully2))

        return self.output(fully3)


class InteractionAleatoricUncertaintyPredictor(InteractionPredictor):
    """
    Interaction layers to estimate Aleatoric uncertainty using the unimodel Gaussian
    output.

    The final variance can be calculated as follows:
        var = log(exp( 1 + X ))
    """
    def __init__(self, input_dim):
        super(InteractionAleatoricUncertaintyPredictor, self).__init__(input_dim)
        self.output = torch.nn.Linear(512, 1)
        self.output_uncertainty = torch.nn.Linear(512, 1)
        torch.nn.init.kaiming_normal_(self.output.weight)

    def forward(self, protein_emb, drug_emb):
        src = torch.cat((protein_emb, drug_emb), 1)
        fully1 = torch.nn.ReLU()(self.fully1(src))
        fully1 = self.dropout(fully1)
        fully2 = torch.nn.ReLU()(self.fully2(fully1))
        fully2 = self.dropout(fully2)
        fully3 = torch.nn.ReLU()(self.fully3(fully2))

        mean = self.output(fully3)
        var = torch.log(torch.exp(self.output_uncertainty(fully3)) + 1)

        return mean, var


class EvidentialLinear(torch.nn.Module):
    """
    *Note* The layer should be putted at the final of the model architecture.

    The output of EvidentialLineary layer is parameters of Normal-Inverse-Gamma (NIG) distribution.
    We can generate the probability distribution of the target value by using the output of this layer.

    The inverse-normal-gamma distribution can be formulated:
    y ~ Normal(mu, sigma**2)
    mu ~ Normal(gamma, (T/nu)**2)
    sigma ~ InverseGamma(alpha, beta)

    where y is a target value such as a durg-target affinity value.

    However, when we train the Evidential network and predict target values by the model,
    we do not directly use the NIG distribution. Our output probability distribution is the distribution
    by analytically marginalizing out mu and sigma[(https://arxiv.org/pdf/1910.02600); equation 6, 7].

    ************************************************************************************************
    *** Target probability distribution:
    *** p(y|gamma, nu, alpha, beta) = t-distribution(gamma, beta*(1+nu)/(nu*alpha) , 2*alpha)
    ***
    *** We can train and infer the true value "y" by using the above probability distribution.
    ************************************************************************************************

    Args:
        gamma(Tensor): The parameter of the NIG distribution. This is the predictive value (predictive mean)
            of the output distribution.
        nu(Tensor): The parameter of the NIG distribution.
        alpha(Tensor): The parameter of the NIG distribution.
        beta(Tensor): The parameter of the NIG distribution.
    """
    def __init__(self, input_dim, output_dim=1):
        """[summary]

        Args:
            input_dim ([type]): [description]
            output_dim (int, optional): [description]. Defaults to 1.
        """
        self.gamma = torch.nn.Linear(input_dim, output_dim)
        self.nu = torch.nn.Linear(input_dim, output_dim)
        self.alpha = torch.nn.Linear(input_dim, output_dim)
        self.beta = torch.nn.Linear(input_dim, output_dim)

    def forward(self, src):
        gamma = self.gamma(src)
        alpha = torch.nn.Softplus()(self.alpha(src)) + 1
        beta = torch.nn.Softplus()(self.beta(src))
        nu = torch.nn.Softplus()(self.nu(src))

        return gamma, alpha, beta, nu


class InteractionEvidentialNetwork(AbstractInteractionModel):
    """
    Deep evidential regression - (https://arxiv.org/pdf/1910.02600)

    Interaction layers using Deep-evidential regression. The output neurons of this network
    are the parameter of Inverse-Normal-Gamma distribution, which is the conjugate prior of Normal.

    The inverse-normal-gamma distribution can be formulated:
    X ~ N(gamma, T/nu)
    T ~ InverseGamma(alpha, beta)

    So we can make t-distribution distribution using the parameters {gamma, nu, alpha, beta}, which are
    the output of this network.
    """
    def __init__(self, input_dim):
        super(InteractionEvidentialNetwork, self).__init__()
        self.fully1 = torch.nn.Linear(input_dim, 1024)
        self.fully2 = torch.nn.Linear(1024, 1024)
        self.fully3 = torch.nn.Linear(1024, 512)

        self.gamma = torch.nn.Linear(512, 1)
        self.nu = torch.nn.Linear(512, 1)#, bias=False)
        self.alpha = torch.nn.Linear(512, 1)#, bias=False)
        self.beta = torch.nn.Linear(512, 1)#, bias=False)

    def forward(self, protein_emb, drug_emb):
        src = torch.cat((protein_emb, drug_emb), 1)
        fully1 = torch.nn.ReLU()(self.fully1(src))
        fully2 = torch.nn.ReLU()(self.fully2(fully1))
        fully3 = torch.nn.ReLU()(self.fully3(fully2))

        gamma = self.gamma(fully3)
        alpha = torch.nn.Softplus()(self.alpha(fully3)) + 1
        beta = torch.nn.Softplus()(self.beta(fully3))
        nu = torch.nn.Softplus()(self.nu(fully3))

        return gamma, nu, alpha, beta


class InteractionEvidentialNetworkDropout(InteractionEvidentialNetwork):
    """[summary]

    Args:
        InteractionEvidentialNetwork ([type]): [description]
    """
    def __init__(self, input_dim):
        super(InteractionEvidentialNetworkDropout, self).__init__(input_dim)
        self.dropout = torch.nn.Dropout(0.1)

    def forward(self, protein_emb, drug_emb):
        src = torch.cat((protein_emb, drug_emb), 1)
        fully1 = self.dropout(torch.nn.ReLU()(self.fully1(src)))
        fully2 = self.dropout(torch.nn.ReLU()(self.fully2(fully1)))
        fully3 = torch.nn.ReLU()(self.fully3(fully2))

        gamma = self.gamma(fully3)
        alpha = torch.nn.Softplus()(self.alpha(fully3)) + 1
        beta = torch.nn.Softplus()(self.beta(fully3))
        nu = torch.nn.Softplus()(self.nu(fully3))

        return gamma, nu, alpha, beta


class InteractionEvidentialNetworkMTL(InteractionEvidentialNetwork):
    """
    Multi-task version of the evidential DTI network.
    The fully-connected modules of the network are divided, one is the
    point-estimation network(fully3) and the other is the bayesian-inference network(fully3_var)

    Methods:
        __init__()

        forward()
    """
    def __init__(self, input_dim):
        super(InteractionEvidentialNetworkMTL, self).__init__(input_dim)
        self.fully3_var = torch.nn.Linear(1024, 512)
        #self.fully2_var = torch.nn.Linear(1024, 1024)
        self.dropout = torch.nn.Dropout(0.1)

    def forward(self, protein_emb, drug_emb):
        src = torch.cat((protein_emb, drug_emb), 1)
        fully1 = self.dropout(torch.nn.ReLU()(self.fully1(src)))
        fully2 = self.dropout(torch.nn.ReLU()(self.fully2(fully1)))
        fully3 = self.dropout(torch.nn.ReLU()(self.fully3(fully2)))
        #fully2_var = self.dropout(torch.nn.ReLU()(self.fully2_var(fully1)))
        fully3_var = self.dropout(torch.nn.ReLU()(self.fully3_var(fully2)))

        gamma = self.gamma(fully3)
        alpha = torch.nn.Softplus()(self.alpha(fully3_var)) + 1
        beta = torch.nn.Softplus()(self.beta(fully3_var))
        nu = torch.nn.Softplus()(self.nu(fully3_var))

        return gamma, nu, alpha, beta



@variational_estimator
class BayesianInteractionPredictor(AbstractInteractionModel):
    """
    Interaction layers to estimate Epistemic uncertainty using Bayesian fully conneceted layers
    using Bayes By Backprop(with blitz library).

    """
    def __init__(self, input_dim):
        super(BayesianInteractionPredictor, self).__init__()
        self.fully1 = BayesianLinear(input_dim, 1024)#, prior_sigma_1=0.1, prior_sigma_2 = 0.0002)
        self.fully2 = BayesianLinear(1024, 1024)#, prior_sigma_1 = 0.1, prior_sigma_2 = 0.0002)
        self.fully3 = BayesianLinear(1024, 512)#, prior_sigma_1 = 0.1, prior_sigma_2 = 0.0002)
        self.output = BayesianLinear(512, 1)#, prior_sigma_1 = 0.1, prior_sigma_2 = 0.0002)


    def forward(self, protein_emb, drug_emb):
        src = torch.cat((protein_emb, drug_emb), 1)
        fully1 = torch.nn.ReLU()(self.fully1(src))
        fully2 = torch.nn.ReLU()(self.fully2(fully1))
        fully3 = torch.nn.ReLU()(self.fully3(fully2))

        return self.output(fully3)



class DeepDTAAleatoricBayes(AbstractDTIModel):
    """
    DeepDTA model with Aleatoric uncertainty modeling using the
    unimodel Gaussian output, which can model the noise(uncertaitny) of data itself.
    """
    def __init__(self, concat_dim=96*2):
        super(DeepDTAAleatoricBayes, self).__init__()
        self.protein_encoder = ProteinEncoder()
        self.smiles_encoder = SMILESEncoder()
        self.interaction_predictor = InteractionAleatoricUncertaintyPredictor(concat_dim)

    def forward(self, d, p):
        """
        Args:
            d(Tensor) : Preprocessed drug input batch
            p(Tensor) : Preprocessed protein input batch
        """
        p_emb = self.protein_encoder(p)
        d_emb = self.smiles_encoder(d)

        return self.interaction_predictor(p_emb, d_emb)


@variational_estimator
class DeepDTAEpistemicBayes(AbstractDTIModel):
    """
    DeepDTA model with Epistemic uncertainty modeling using the
    Bayesian neural network layer, which can model the uncertainty of model parameters(posteriors).
    """
    def __init__(self, concat_dim=96*2):
        super(DeepDTAEpistemicBayes, self).__init__()
        self.protein_encoder = BayesianProteinEncoder()
        self.smiles_encoder = BayesianSMILESEncoder()
        self.interaction_predictor = BayesianInteractionPredictor(concat_dim)

    def forward(self, d, p, sample_nbr=10):
        """
        Args:
            input_(tuple) -> (d, p)
                d(Tensor) : Preprocessed drug input batch
                p(Tensor) : Preprocessed protein input batch
        """
        p_emb = self.protein_encoder(p)
        d_emb = self.smiles_encoder(d)

        return self.interaction_predictor(p_emb, d_emb)

    def mfvi_forward_dti(self, d, p, sample_nbr=10):
        """
        Overroaded original method from blitz library

        Performs mean-field variationalinference for the variational estimator model:
            Performs sample_nbr forward passes with uncertainty on the weights, returning its mean and standard deviation
        Parameters:
            inputs: torch.tensor -> the input data to the model
            sample_nbr: int -> number of forward passes to be done on the data
        Returns:
            mean_: torch.tensor -> mean of the perdictions along each of the features of each datapoint on the batch axis
            std_: torch.tensor -> std of the predictions along each of the features of each datapoint on the batch axis
        """
        result = torch.stack([self(d, p) for _ in range(sample_nbr)])
        return result.mean(dim=0), result.std(dim=0)

    def sample_elbo_cus(self, d, p, labels, criterion, sample_nbr=5, complexity_cost_weight=1):
        loss = 0
        for _ in range(sample_nbr):
            outputs = self(d, p)
            loss += criterion(outputs, labels)
            loss += self.nn_kl_divergence() * complexity_cost_weight
        return loss/sample_nbr


@variational_estimator
class DeepDTAEpistemicBayesStudent(DeepDTAEpistemicBayes):
    """
    DeepDTA model with Epistemic uncertainty modeling using the
    Bayesian neural network layer, which can model the uncertainty of model parameters(posteriors).
    """
    def __init__(self, concat_dim=96*2):
        super(DeepDTAEpistemicBayesStudent, self).__init__(concat_dim=concat_dim)
        self.interaction_predictor = BayesianInteractionPredictor(concat_dim)

    def forward(self, d, p, sample_nbr=10):
        """
        If sample_nbr == 1, it is just one-time prediction with weight sampling.

        Args:
            d(Tensor) : Preprocessed drug input batch
            p(Tensor) : Preprocessed protein input batch

        """
        p_emb = self.protein_encoder(p)
        d_emb = self.smiles_encoder(d)

        predictions = torch.stack([self.interaction_predictor(p_emb, d_emb) for _ in range(sample_nbr)]) # S*N*1
        mean = predictions.mean(dim=0) # N*1
        var = torch.sum((predictions - mean)**2, dim=0) / (sample_nbr - 1)
        std = var**0.5

        # ISSUE: prediction.std(dim=0) raise a error.
        # prediction.mean(dim=0) did not raise a error
        return mean, std#torch.std(predictions, dim=0)


class EvidentialDeepDTA(DeepDTA):
    """
    DeepDTA model with Prior interaction networks.

    """
    def __init__(self, concat_dim=96*2, dropout=True, mtl=False):
        super(EvidentialDeepDTA, self).__init__(concat_dim=concat_dim)
        if dropout:
            self.interaction_predictor = InteractionEvidentialNetworkDropout(concat_dim)
            if mtl:
                self.interaction_predictor = InteractionEvidentialNetworkMTL(concat_dim)
        else:
            self.interaction_predictor = InteractionEvidentialNetwork(concat_dim)

    def forward(self, d, p):
        output_tensors = super().forward(d, p)

        return output_tensors

    @staticmethod
    def aleatoric_uncertainty(nu, alpha, beta):
        return torch.sqrt(beta/(alpha-1))

    @staticmethod
    def epistemic_uncertainty(alpha, beta):
        return torch.sqrt(beta/(nu*(alpha-1)))

    @staticmethod
    def total_uncertainty(nu,alpha,beta):
        """
        Return standard deviation of Generated student t distribution,

        p(y|gamma, nu, alpha, beta)  = Student-t(y; gamma, beta*(1+nu)/(nu*alpha), 2*alpha )

        Note that the freedom of given student-t distribution is 2alpha.
        """
        return torch.sqrt(beta*(1+nu)/(nu*alpha))

    def predictive_entropy(self, nu, alpha, beta):
        scale = (beta*(1+nu)/(nu*alpha)).sqrt()
        df = 2*alpha
        dist = studentT.StudentT(df = df, scale=scale)
        return dist.entropy()

    @staticmethod
    def freedom(alpha):
        return 2*alpha


class DeepDTAPriorNetwork(EvidentialDeepDTA):
    """
    Class to load old models
    Args:
        EvidentialDeepDTA ([type]): [description]
    """
    def __init__(self, concat_dim=96*2, dropout=True, mtl=False):
        super(DeepDTAPriorNetwork, self).__init__(concat_dim=concat_dim, dropout=dropout, mtl=mtl)
    def forward(self, d, p):
        return super(DeepDTAPriorNetwork, self).forward(d, p)

class InteractionPriorNetworkDropout(InteractionEvidentialNetwork):
    def __init__(self, input_dim):
        super(InteractionPriorNetworkDropout, self).__init__(input_dim)

    def forward(self, protein_emb, drug_emb):
        return super(InteractionPriorNetworkDropout, self).forward(protein_emb, drug_emb)


def activation_similarity(y, labels=None):
    if labels == None:
        permute = torch.randperm(y.size(0))
        y_mixed = y[permute].to(y.device)
        sim = torch.square( F.cosine_similarity(y.view(y.size(0), -1), y_mixed.view(y.size(0), -1)) ).mean()
    else:
        possible_labels = torch.unique(labels)
        sim = 0
        for l_i in possible_labels:
            idx = (labels == l_i).nonzero()
            rand_idx = idx[torch.randperm(len(idx))]
            sim += torch.square( F.cosine_similarity(y[idx].view(y[idx].size(0), -1), y[rand_idx].view(y[rand_idx].size(0), -1)) ).sum()/len(idx)
    return sim
