const { logExpression, setLogLevel } = require('@cisl/zepto-logger');
const envLoaded = require("dotenv").config({ silent: true });
if (!envLoaded) console.log("warning:", __filename, ".env cannot be found");
const http = require("http");
const path = require("path");
const fs = require("fs");
const csv = require('csv-parser')
const {retrieveRecords} = require('./database-functions.js');

let logLevel = 2;
setLogLevel(logLevel);

const express = require('express');
const methodOverride = require('method-override');
const cors = require('cors');

let myPort = 9857;

//let asset_graph = 'asset_graph_full_20250305.json';
//asset_graph = 'asset_graph_14_objects_full.20250707.json';
let asset_graph = 'abstract_graph_14_objects.data.fk.json';
let defaultAutoForeignKeyInference = false;
let assetAllowedTables = ["ASSET", "WORKORDER", "SR", "PO", "LABOR", "JOBPLAN", "ITEM", "PM", "COMPANIES", "MATUSETRANS", "LABTRANS", "ASSIGNMENT", "ASSETMETER"];

process.argv.forEach((val, index, array) => {
  if (val === '--port') {
    myPort = array[index + 1];
  }
  else if (val === '--level') {
    logLevel = array[index + 1];
    logExpression('Setting log level to ' + logLevel, 1);
  } 
  else if (val === '--asset_graph') {
    asset_graph = array[index + 1];
    logExpression('Setting log level to ' + asset_graph, 1);
  }
  else if (val === '--defaultAutoForeignKeyInference') {
    defaultAutoForeignKeyInference = array[index + 1];
    logExpression('Setting defaultAutoForeignKeyInference to ' + defaultAutoForeignKeyInference, 1);
  }
});

setLogLevel(logLevel);

function getSafe(p, o, d) {
  return p.reduce((xs, x) => (xs && xs[x] != null && xs[x] != undefined) ? xs[x] : d, o);
}

const app = express();
app.use(cors());
app.set('port', process.env.PORT || myPort);
// app.use(bodyParser.json());
// app.use(bodyParser.urlencoded({ extended: true, limit: '50mb'}));
// app.use(methodOverride());

app.use(methodOverride());
app.use(express.json({limit: '50mb'}));
app.use(express.urlencoded({extended: true, limit: '50mb'}));

// Provide utterance, sql, and either schemaFile or schemaJSON file names as input
// Using mongo parsing
app.post('/generateRandomInstantiation', (req, res) => {
  logExpression("Called /generateRandomInstantiation.", 2);
  let useFakeValueRange = getSafe(['useFakeValueRange'], req.body, false);
  let abstract_graph = getSafe(['abstract_graph'], req.body, null);
  logExpression("Got abstract_graph: ", 2);
  logExpression(abstract_graph, 2);
  let target_graph = getSafe(['target_domain_abstract_graph'], req.body, null);
  let target_graph_file = getSafe(['target_graph_file'], req.body, asset_graph);
  let allowedTables = getSafe(['allowedTables'], req.body, null);

  if(!target_graph && target_graph_file) {
    target_graph = require(path.join(process.env.PWD, target_graph_file));
  }

  if(target_graph) {
    logExpression("Going to call transmogrify.", 2);
    target_graph = transmogrifyGraph(target_graph, allowedTables);
    logExpression("Transmogrified graph is: ", 2);
    logExpression(target_graph, 2);
  }
  //let target_graph_file = getSafe(['target_graph_file'], req.body, 'asset_graph_full_tipu.20250130.json');
  let useForeignKeyInference = getSafe(['useForeignKeyInference'], req.body, defaultAutoForeignKeyInference);

  let instances = getSafe(['instances'], req.body, 1);
  let generate_values = getSafe(['generate_values'], req.body, false);
  logExpression("Received input json with abstract_graph: ", 2);
  logExpression(abstract_graph, 2);


  if(!useFakeValueRange) {
    logExpression("Pruning target_graph.", 2);
    target_graph = pruneTargetGraph(target_graph);
  }

  logExpression("target_graph is: ", 3);
  logExpression(target_graph, 3);

  if(useForeignKeyInference) {
    abstract_graph = addAutoForeignKeys(abstract_graph);
    logExpression("After adding automatic foreign keys, abstract graph is: ", 2);
    logExpression(abstract_graph, 2);
  }

  let results = generateRandomInstantiation(abstract_graph, target_graph, instances, generate_values);

  res.json(results);
});

app.post('/batchTest', (req, res) => {
  logExpression("Called /batchTest.", 2);
  let useFakeValueRange = getSafe(['useFakeValueRange'], req.body, false);
  let batch_file = getSafe(['batch_file'], req.body, 'parsed_log.json');
  let target_graph = getSafe(['target_graph'], req.body, null);
  let target_graph_file = getSafe(['target_graph_file'], req.body, asset_graph);
  let instances = getSafe(['instances'], req.body, 1);
  let generate_values = getSafe(['generate_values'], req.body, false);
  let useForeignKeyInference = getSafe(['useForeignKeyInference'], req.body, defaultAutoForeignKeyInference);
  if(!target_graph && target_graph_file) {
    target_graph = require(path.join(process.env.PWD, target_graph_file));
  }

  if(!useFakeValueRange) {
    target_graph = pruneTargetGraph(target_graph);
  }

  logExpression("target_graph is: ", 3);
  logExpression(target_graph, 3);

  let batch = require(path.join(process.env.PWD, batch_file));

  let taskList = batch.map(el => {
    let ag = JSON.parse(JSON.stringify(el.service_request.abstract_graph));
    if(useForeignKeyInference) {
      ag = addAutoForeignKeys(ag);
    }
    return {
      qNumber: el.qNumber,
      source_sql: el.source_sql,
      templatized_sql: el.templatized_sql,
      abstract_graph: ag,
      target_graph: target_graph,
      instances: instances,
      generate_values: generate_values,
      error: el.error
    };
  });
  logExpression("taskList has length " + taskList.length, 2);

  //taskList = taskList.slice(0,4); // Use this when testing to shorten cycle time

  logExpression("taskList: ", 3);
  logExpression(taskList, 3);
  let results = [];
  let numErrors = 0;
  let numFailures = 0;
  let numEF = 0;

  taskList.forEach((task, i) => {
    logExpression("\nProcessing task #" + i, 2);
    let riResult = generateRI(task);
    if(!riResult.result || !riResult.result.length) {
      riResult.success = false;
      numFailures++;
    }
    else {
      riResult.success = true;
    }
    if(riResult.error) {
      numErrors++;
      if(!riResult.success) {
        numEF++;
      }
    }
    results.push(riResult);
  });

  logExpression("Failures: " + numFailures, 2);
  logExpression("Reported errors: " + numErrors, 2);
  logExpression("Reported error and failure: " + numEF, 2);

  res.json(results);
});

app.get('/produceEvaluationFiles', (req, res, next) => {
  let query = {
    "ValidatedByHuman": "accept"
  }

  let infile = 'asset.spider.dev.20250720.validated.regenerated.rows.csv';
  let fullfile = path.join(process.env.PWD, "ui", infile);
  logExpression(fullfile, 2);
  let original_records = {};
  fs.createReadStream(fullfile)
  .pipe(csv())
  .on('data', (data) => {
    logExpression(data, 2);
    let QuestionID = data.SourceDB + '.' + data.SourceDBIndex;
    original_records[QuestionID] = {
      QuestionID: QuestionID,
      TargetQuestionOriginal: data.TargetQuestion,
      TargetSQLOriginal: data.TargetSQL
    }
  })
  .on('end', () => {
    logExpression("Finished reading in original records: ", 2);
    logExpression(original_records, 2);

    let outFile = path.join(process.env.PWD, 'ui', 'gold_evaluation.json');

    return retrieveRecords(query)
    .then(records => {
      records = records.map(record => {
        let new_record = JSON.parse(JSON.stringify(record));
        let QuestionID = record.QuestionID;
        new_record.TargetQuestionOriginal = original_records[QuestionID].TargetQuestionOriginal;
        new_record.TargetSQLOriginal = original_records[QuestionID].TargetSQLOriginal;
        new_record.SourceDB = record.sourceDBName
        delete new_record.sourceDBName;
        delete new_record._id;
        delete new_record.__v;
        return new_record;
      });

      logExpression("About to publish to outFile " + outFile, 2);
      logExpression("About to compute JSON.stringify of out.", 2);

      let outString = JSON.stringify(records, null, 2);

      logExpression("outString: ", 2);
      logExpression(outString, 2);

      fs.writeFileSync(outFile, outString);
      res.json({msg: "All done"});
    })
    .catch(e => {
      logExpression("ERROR: ", 1);
      logExpression(e, 1);
      res.json({msg: "ERROR"});
    })
  })
});

app.use(function(err, req, res, next) {
    logExpression('Something broke!  Please try again.', 1);
    logExpression(err, 2);
    res.status(500).send('Something broke!  Please try again. \n' + JSON.stringify(err,null,2));
});

// Set up and initialize the server
const server = http.createServer(app);
server.listen(app.get('port'), () => {
  logExpression('Express server listening on port ' + app.get('port'), 1);
});

function generateRandomInstantiation(abstract_graph, target_graph, instances, generate_values) {
  logExpression("Entered generateRandomInstantiation.", 2);
  logExpression(abstract_graph, 3);
  logExpression(target_graph, 3);
  logExpression(instances, 3);
  logExpression(generate_values, 3);

  // Normalize varchar to be "string"
  abstract_graph.entities = abstract_graph.entities.map(entityBlock => {
    let eBlock = JSON.parse(JSON.stringify(entityBlock));
    if(eBlock.dataType && eBlock.dataType.toLowerCase().includes("varchar")) {
      eBlock.dataType = "text";
    }
    return eBlock;
  });

  Object.keys(target_graph).forEach(key => {
    logExpression("key: " + key, 2);
    Object.keys(target_graph[key]).forEach(key2 => {
      delete target_graph[key][key2].primaryKeys;
    });
  });

  logExpression("Now target_graph is: ", 3);
  logExpression(target_graph, 3);

  let results = [];
  let failures = 0;

  do {
    let graph_info;
    let graphOK;
    let graphUnique;
    let tries = 0;
    do { // Attempt to generate a legitimate, unique instance
      let ag = JSON.parse(JSON.stringify(abstract_graph));
      let tg = JSON.parse(JSON.stringify(target_graph));
      graph_info = generateRandomIsomorphicGraph(ag, tg, generate_values);
      logExpression("Before checkGraph.", 2);
      graphOK = checkGraph(graph_info.agraph_dictionary);
      logExpression("graphOK: " + graphOK, 2);
      logExpression("Just tested graph dictionary: ", 3);
      logExpression(graph_info.agraph_dictionary, 3);
      if(graphOK) {
        let indx = 0;
        graphUnique = true;
        while(graphUnique && results.length && indx < results.length) {
          let result = results[indx];
          let match = matchDict(graph_info.agraph_dictionary, result.abstract_to_target);
          graphUnique = graphUnique && !match;
          indx++;
        };
        logExpression("graphUnique: " + graphUnique, 2);
      }
      tries++;
    }
    while (!(graphOK && graphUnique) && tries < 10);

    if(!graphOK || !graphUnique) {
      failures++;
    }
    else {
      let result = {
        attempts: tries,
        success: graphOK,
        abstract_to_target: graph_info.agraph_dictionary,
        target_to_abstract: graph_info.rgraph_dictionary
      }

      results.push(result);
      logExpression("Now there are " + results.length + " instances.", 2);

    }
  }
  while (results.length < instances && failures < 10);

  let target_graph_dict = {};
  target_graph.entities.forEach(node => {
    target_graph_dict[node.id] = node.description;
  });
  results = addDescriptions(results, target_graph_dict);

  results = addRules(results);

  logExpression("Just before returning the JSON, results are: ", 3);
  logExpression(results, 3);
  logExpression("About to return " + results.length + " results from /generateRandomInstantiation.", 3);
  return results;
}

function addRules(graph) {
  logExpression("Just got in addRules.", 3);
  let domain_rules = require('./domain_rule_dictionary.json');

  let newgraph = [];
  graph.forEach(el => {
    let newEl = JSON.parse(JSON.stringify(el));
    let att = el.abstract_to_target;
    logExpression("Checking to see if there is a relevant rule for: ", 3);
    logExpression(att, 3);
    Object.keys(att).forEach(label => {
      logExpression("Considering " + label, 3);
      if(att.type == "value") {
        let table = att.target_table;
        let attribute = att.target_column.split('.')[1];
        let value = att.target_name;
        logExpression("considering table, attribute, value: ", 2);
        logExpression(table, 2);
        logExpression(attribute, 2);
        logExpression(value, 2);

        if(domain_rules[table][attribute][value]) {
          logExpression("Adding a substitution rule.", 2);
          let substitution_rule = {
            rulename: domain_rules[table][attribute][value].rulename,
            translation: domain_rules[table][attribute][value].translation
          }
          att.substitution_rule = substitution_rule;
        }
        else {
          logExpression("Not adding a substitution rule.", 2);
        }
      }
    });
    newEl.abstract_to_target = att;
    newgraph.push(newEl);
  });
  return newgraph;
}

function addDescriptions(mappings, target_graph_dict) {
  let newmappings = mappings.map(mBlock => {
    let newBlock = JSON.parse(JSON.stringify(mBlock));
    let att = mBlock.abstract_to_target;
    Object.keys(att).forEach(key => {
      let description = target_graph_dict[att[key].target_id];
      if(description) {
        newBlock.abstract_to_target[key].target_description = description;
      }
    });
    return newBlock;
  });
  return newmappings;
}

function matchDict(dict1, dict2) {

  let match = true;
  if(Object.keys(dict1).length != Object.keys(dict2).length) {
    return false;
  }
  Object.keys(dict1).forEach(key => {
    if(!dict2[key]) {
      match = false;
    }
    else {
      if(dict1[key].type != dict2[key].type) {
        match = false;
      }
      if(dict1[key].type != "value") {
        match = match && (dict1[key].target_id == dict2[key].target_id);
      }
    }
  });
  return match;
}

// Takes as input an abstract graph agraph representing the key elements of a templatized SQL and
// a real graph rgraph representing the target benchmark domain
// Generates a forward and reverse dictionary with mappings from abstract domain to real domain
function generateRandomIsomorphicGraph(agraph, rgraph, generate_values) {
  logExpression("Entered generateRandomIsomorphicGraph.", 2);

  // Sort source relationships to bring foreignKey first
  // Second priority is to bring relationships involving column -> value to the top (after foreignKey relationships)
  
  let abstract_relationships = [];

  let agraph_dictionary = {};
  let rgraph_dictionary = {};

  if(agraph.relationships.length) {
    abstract_relationships = agraph.relationships.sort((a,b) => {
      if(a.relationship == "foreignKey" && b.relationship != "foreignKey") {
        return -1;
      }
      else if (a.target.split('.').length > 2) {
        return -1;
      }
      else { 
        return 0;
      }
    });
  

    // Initialize agraph_dict by taking the agraph.entities array
    // and turning it into a dictionary with keys = the id field of each entity
    // This structure will be augmented to include a target_id field that indicates the
    // selected element in the real target domain
    agraph.entities.forEach(entityBlock => {
      agraph_dictionary[entityBlock.id] = entityBlock;
    });

    // Initialize rgraph_dict by taking the rgraph.entities array
    // and turning it into a dictionary with keys = the id field of each entity
    // This structure will be augmented to include an abstract_id field that indicates the
    // selected element in the abstract source domain
    rgraph.entities.forEach(entityBlock => {
      rgraph_dictionary[entityBlock.id] = entityBlock;
    });

    // Loop over all relationships in abstract graph; this should cover all entities
    logExpression("Loop over " + abstract_relationships.length + " abstract_relationships.", 2);
    abstract_relationships.forEach(aRelation => { // The relationship is of type foreignKey
      logExpression("Considering aRelation: ", 3);
      logExpression(aRelation, 3);
      if(aRelation.relationship == "foreignKey") {
        logExpression("Relationship is foreignKey.", 3);
        let fKResults = mapForeignKeys(aRelation, agraph, agraph_dictionary, rgraph, rgraph_dictionary);
        agraph_dictionary = fKResults.agraph_dictionary;
        rgraph_dictionary = fKResults.rgraph_dictionary;
      }
      else if (aRelation.relationship == "parent") { // The relationship is of type parent
        logExpression("Relationship is parent.", 3);
        logExpression(agraph_dictionary, 3);
        let type = agraph_dictionary[aRelation.source].type;
        logExpression("type: " + type, 3);
        let aValue, aColumn, aTable;

        if(type == "value") { // If this is an abstract value->column relationship, get the value, column and table
          aValue = aRelation.source;
          aColumn = getParent(aValue, agraph.relationships);
          aTable = getParent(aColumn, agraph.relationships);
        }

        else if (type == "column") { // If this is an abstract column->parent relationship, get the column and table
          logExpression("Type is column.", 3);
          aColumn = aRelation.source; // Abstract column
          logExpression("About to getParent.", 3);
          logExpression(aColumn, 3);
          logExpression(agraph.relationships, 3);
          aTable = getParent(aColumn, agraph.relationships); // Abstract table
        }

        // Retrieve the real table if it has already been chosen; if not then select a valid table
        // that satisfies all relevant constraints
        logExpression("About to call selectRealTable.", 3);
        let table_selection_results = selectRealTable(aTable, agraph_dictionary, rgraph, rgraph_dictionary);
        let rTable = table_selection_results.rTable;
        agraph_dictionary = table_selection_results.agraph_dictionary;
        rgraph_dictionary = table_selection_results.rgraph_dictionary;

        // Map the abstract column to a real column, and also create a mapping from the abstract table
        // to the real one if it doesn't already exist
        let column_selection_results = selectRealColumn(aColumn, rTable, agraph_dictionary, rgraph, rgraph_dictionary);
        agraph_dictionary = column_selection_results.agraph_dictionary;
        rgraph_dictionary = column_selection_results.rgraph_dictionary;

        if (type == "value") {
          // Map the abstract value to a real one, given all constraints established so far.
          agraph_dictionary = selectRealValue(aValue, aColumn, agraph_dictionary, rgraph, rgraph_dictionary, generate_values);
        }
      }
    });
  }
  else { // Must just be tables with no edges
    logExpression("There were no relationships.", 2);

    // Initialize agraph_dict by taking the agraph.entities array
    // and turning it into a dictionary with keys = the id field of each entity
    // This structure will be augmented to include a target_id field that indicates the
    // selected element in the real target domain
    agraph.entities.forEach(entityBlock => {
      agraph_dictionary[entityBlock.id] = entityBlock;
    });

    // Initialize rgraph_dict by taking the rgraph.entities array
    // and turning it into a dictionary with keys = the id field of each entity
    // This structure will be augmented to include an abstract_id field that indicates the
    // selected element in the abstract source domain
    rgraph.entities.forEach(entityBlock => {
      rgraph_dictionary[entityBlock.id] = entityBlock;
    });

    logExpression(rgraph_dictionary, 3);
    agraph.entities.forEach(aEntityBlock => {
      logExpression("aEntityBlock: ", 3);
      logExpression(aEntityBlock, 3);
      if(!aEntityBlock.target_id) {
        logExpression("About to call selectRealTable.", 3);
        logExpression(agraph_dictionary, 3);
        logExpression(rgraph_dictionary, 3);
        logExpression(rgraph, 3);
        let aTable = aEntityBlock.id;
        logExpression("aTable: " + aTable, 3);
        let table_selection_results = selectRealTable(aTable, agraph_dictionary, rgraph, rgraph_dictionary);
        logExpression("table_selection_results: ", 3);
        logExpression(table_selection_results, 3);
        agraph_dictionary = table_selection_results.agraph_dictionary;
        rgraph_dictionary = table_selection_results.rgraph_dictionary;
      }
    });
  }

  let pruned_rgraph_dictionary = {};
  logExpression("Now pruning rgraph.", 2);
  Object.keys(rgraph_dictionary).forEach(key => {
    if(rgraph_dictionary[key].abstract_id) {
      pruned_rgraph_dictionary[key] = rgraph_dictionary[key];
    }
  });
  logExpression("About to return results from generateRandomIsomorphicGraph.", 2);
  return {agraph_dictionary, rgraph_dictionary: pruned_rgraph_dictionary};
}

function matchDataType(abstractInfo, realInfo) {
  logExpression("In matchDataType with abstractInfo and realInfo: ", 3);
  logExpression(abstractInfo, 3);
  logExpression(realInfo, 3);
  let match = false;
  if(!abstractInfo.dataType) {
    match = true;
  }
  if (abstractInfo.dataType == realInfo.dataType) {
    match = true;
  }
  if (abstractInfo.dataType == "integer" && realInfo.dataType == "text") {
    match = true;
  }
  if (abstractInfo.dataType == "text" && realInfo.dataType == "integer") {
    match = true;
  }

  return match;
}

function matchID(abstractInfo, realID) {
  return !abstractInfo.target_id || abstractInfo.target_id == realID;
}

function getParent(id, relationships) {
  logExpression("In getParent with id " + id, 3);
  if(!relationships) {
    return null;
  }
  let parents = relationships.filter(rel => {
    return rel.source == id && rel.relationship == "parent";
  });
  if (parents.length == 0) {
    return null;
  }
  if (parents.length == 1) {
    return parents[0].target;
  }
  if (parents.length > 1) {
    logExpression("WARNING: Found " + parents.length + " parents of node with id = " + id, 2);
    return parents[0].target;
  }
}

function mapForeignKeys(aRelation, agraph, agraph_dictionary, rgraph, rgraph_dictionary) {
  logExpression("In mapForeignKeys.", 2);
  logExpression("aRelation: ", 3);
  logExpression(aRelation, 3);
  logExpression(agraph, 3);
  logExpression(agraph_dictionary, 3);
  logExpression(rgraph_dictionary, 3);
  logExpression("rgraph: ", 3);
  logExpression(rgraph, 3);

  // The relationship is a foreignKey relation.
  // Loop over all source-target pairs in the real domain
  // Filter candidate source-target pairs to those with types and table parents analogous to those in abstract domain
  // Eliminate any real candidate sources or targets that have already been selected
  let candidates = rgraph.relationships.filter(rRelation => {
    logExpression("rRelation: ", 3);
    logExpression(rRelation, 3);
    
    // Only consider real candidates with foreign key relationships
    let pass = (rRelation.relationship == "foreignKey");
    logExpression("After fk, pass is " + pass, 3);

    // If the abstract source aRelation.source has already been mapped to a real source node,
    // ensure that we only consider source-target pairs with this specific real source node
    pass = pass && matchID(agraph_dictionary[aRelation.source], rRelation.source);
    logExpression("After 2, pass is " + pass, 3);


    // If data type matters in the abstract graph, restrict based on data type
    pass = pass && matchDataType(agraph_dictionary[aRelation.source], rgraph_dictionary[rRelation.source]);
    logExpression("After 3, pass is " + pass, 3);



    // If the abstract target aRelation.target has already been mapped to a real target node,
    // ensure that we only consider source-target pairs with this specific real target node
    pass = pass && matchID(agraph_dictionary[aRelation.target], rRelation.target);
    logExpression("After 4, pass is " + pass, 3);



    // If data type matters in the abstract graph, restrict based on data type
    pass = pass && matchDataType(agraph_dictionary[aRelation.target], rgraph_dictionary[rRelation.target]);
    logExpression("After 5, pass is " + pass, 3);


    // Determine the table (parent) with which the source and target are associated in the real domain
    let rTableSourceID = getParent(rRelation.source, rgraph.relationships);
    let rTableTargetID = getParent(rRelation.target, rgraph.relationships);

    // Check whether the source table has already been mapped; if so enforce that constraint
    let abstract_id_source = getSafe([rTableSourceID, "abstract_id"], rgraph_dictionary, null);
    if(abstract_id_source) { // The sourceTable has already been pinned at a value; enforce this constraint
      pass = pass && abstract_id_source == getParent(aRelation.source, agraph.relationships);
    }
    logExpression("After 6, pass is " + pass, 3);


    // Check whether the target table has already been mapped; if so enforce that constraint
    let aidTarget = getSafe([rTableTargetID, "abstract_id"], rgraph_dictionary, null);
    if(aidTarget) { // The targetTable has already been pinned at a value
      pass = pass && aidTarget == getParent(aRelation.target, agraph.relationships);
    }

    logExpression("After 7, pass is " + pass, 3);


    return pass;
  });

  logExpression("There are " + candidates.length + " candidates for aRelation.", 3);

  // All of the remaining candidates are valid. Select one of them randomly.
  let candidate = selectRandom(candidates);

  logExpression("Selected candidate foreignKey pair: ", 3);
  logExpression(candidate, 3);

  if(!candidate) {
    return {agraph_dictionary, rgraph_dictionary};
  }

  else {
    // Now that candidate has been selected, add the target_id to agraph_dictionary for the column and the associated tables if necessary.
    if(!agraph_dictionary[aRelation.source].target_id) {
      agraph_dictionary[aRelation.source].target_id = candidate.source;
      agraph_dictionary[aRelation.source].target_name = rgraph_dictionary[candidate.source].name;

      rgraph_dictionary[candidate.source].abstract_id = aRelation.source;
      let aTableArray = aRelation.source.split('.');
      let aTable = aTableArray[0];
      let rTableArray = candidate.source.split('.');
      let rTable = rTableArray[0];
      agraph_dictionary[aRelation.source].target_table = rTable; // Need to add this to fix bug noted by Tipu 2025-04-11

      if(!agraph_dictionary[aTable].target_id) {
        agraph_dictionary[aTable].target_id = rTable;
        agraph_dictionary[aTable].target_name = rgraph_dictionary[rTable].name;
        rgraph_dictionary[rTable].abstract_id = aTable;
      }
    }
    if(!agraph_dictionary[aRelation.target].target_id) {
      agraph_dictionary[aRelation.target].target_id = candidate.target;
      agraph_dictionary[aRelation.target].target_name = rgraph_dictionary[candidate.target].name;

      rgraph_dictionary[candidate.target].abstract_id = aRelation.target;
      let aTableArray = aRelation.target.split('.');
      let aTable = aTableArray[0];
      let rTableArray = candidate.target.split('.');
      let rTable = rTableArray[0];
      agraph_dictionary[aRelation.target].target_table = rTable; // Need to add this to fix bug noted by Tipu 2025-04-11

      if(!agraph_dictionary[aTable].target_id) {
        agraph_dictionary[aTable].target_id = rTable;
        agraph_dictionary[aTable].target_name = rgraph_dictionary[rTable].name;
        rgraph_dictionary[rTable].abstract_id = aTable;
      }
    }
  
    return {agraph_dictionary, rgraph_dictionary};
  }
}

function selectRealTable(aTable, agraph_dictionary, rgraph, rgraph_dictionary) {
  logExpression("Just got in selectRealTable.", 3);
  let rTable;
  if(!agraph_dictionary[aTable].target_id) { // table id not yet assigned
    logExpression("table id not yet assigned.", 2);
    let candidates = rgraph.entities.filter(entityBlock => {
      return entityBlock.type == "table" && !rgraph_dictionary[entityBlock.id].abstract_id
    });
    let candidate = selectRandom(candidates);
    if(candidate) {
      agraph_dictionary[aTable].target_id = candidate.id;
      agraph_dictionary[aTable].target_name = rgraph_dictionary[candidate.id].name;
      rgraph_dictionary[candidate.id].abstract_id = aTable;
      rTable = candidate.id;
    }
    else {
      logExpression("No candidates found.", 2);
      rTable = null;
    }
  }
  else { // Table id already assigned; establish this as a constraint
    rTable = agraph_dictionary[aTable].target_id;
  }
  return {rTable, agraph_dictionary, rgraph_dictionary};
}

function selectRealColumn(aColumn, rTable, agraph_dictionary, rgraph, rgraph_dictionary) {
  let rColumn;
  let dataType = safeLowerCase(agraph_dictionary[aColumn].dataType);

  if(!agraph_dictionary[aColumn].target_id) { // column id is not yet assigned

    // We are here because the column id is not yet mapped.
    // Loop over all entities in the real graph and exclude those that fail any pertinent criteria
    let candidates = rgraph.entities.filter(entityBlock => {

      // Ensure that the entity is of type column and hasn't already been selected
      let pass = entityBlock.type == "column" && !rgraph_dictionary[entityBlock.id].abstract_id;

      // Ensure that the table of the column in the proposed match has the correct table
      pass = pass && getParent(entityBlock.id, rgraph.relationships)  == rTable;

      // If data type matters, make sure it is correct
      pass = pass && (!dataType || dataType == entityBlock.dataType);

      return pass;
    });

    // Randomly select one of the valid candidates
    let candidate = selectRandom(candidates);
    logExpression("randomly selected candidate: ", 3);
    logExpression(candidate, 3);

    // If a candidate was selected, update the agraph and rgraph dictionaries accordingly
    if(candidate) {
      logExpression("candidate was selected.", 3);
      logExpression(candidate, 3);
      agraph_dictionary[aColumn].target_id = candidate.id;
      agraph_dictionary[aColumn].target_name = rgraph_dictionary[candidate.id].name;
      agraph_dictionary[aColumn].target_table = rTable;
      rgraph_dictionary[candidate.id].abstract_id = aColumn;
      rColumn = candidate.id;
      logExpression("rColumn: " + rColumn, 2);
    }
    else {
      logExpression("No candidates found in selectRealColumn.", 2);
      rColumn = null;
    }
    return {rColumn, agraph_dictionary, rgraph_dictionary};
  }
  else { // target column has already been established; set this as constraint
    rColumn = agraph_dictionary[aColumn].target_id;
  }
  return {rColumn, agraph_dictionary, rgraph_dictionary};
}

function selectRealValue(aValue, aColumn, agraph_dictionary, rgraph, rgraph_dictionary, generate_values) {
  logExpression("Entered selectRealValue.", 2);
  if(!agraph_dictionary[aValue].target_id) { // value id not yet assigned; if not true then nothing to do

    let column_target_id = agraph_dictionary[aColumn].target_id;
    if(column_target_id) {
        if(generate_values) {
            let randomValue = generateRandomValue(rgraph_dictionary[column_target_id]);
            agraph_dictionary[aValue].target_id = randomValue;
            agraph_dictionary[aValue].target_name = randomValue;
        }
        agraph_dictionary[aValue].target_column = column_target_id;
        agraph_dictionary[aValue].target_table = getParent(column_target_id, rgraph.relationships);
    }
    else {
      logExpression("Did not find a column in selectRealValue -- cannot compute a value for " + aColumn + ", " + aValue, 2);
    }
  }
  return agraph_dictionary;
}

function checkGraph(dict) {
  let pass = true;
  Object.keys(dict).forEach(key => {
    let ok = (dict[key].target_id != undefined) && (dict[key].target_id != null);
    ok = ok || dict[key].type == "value" || dict[key].type == "created"; // We don't need to fill in values.
    pass = pass && ok;
  });
  return pass;
}

let defaultValueRange = {
  number: [0, 10],
  integer: [0, 10],
  real: [0, 10],
  date: ["2000-01-01", "2024-12-31"],
  string: ["A", "B", "C", "D"]
};

let useDefaultValueRanges = true;

function generateRandomValue(infoBlock) {
  logExpression("In generateRandomValue with infoBlock: ", 3);
  logExpression(infoBlock, 3);
  let dataType = safeLowerCase(infoBlock.dataType);
  let valueRange = infoBlock.valueRange;
  if(!infoBlock.valueRange) {
    logExpression("No value range! Using defaults.", 1);
    if(useDefaultValueRanges) {
      valueRange = defaultValueRange[dataType] || defaultValueRange.string;
      logExpression("Now valueRange is: ", 2);
      logExpression(valueRange, 2);
    }
    else {
      logExpression("Unable to fill in a value; couldn't establish a default range.", 1);
      return null;
    }
  }
  logExpression("Now try to fill in a value with dataType: " + dataType + ".", 3);
  let nonEnumerableTypes = ["number", "integer", "real", "date"];

  if(nonEnumerableTypes.includes(dataType)) {
    if(dataType != "date") {
      let value = valueRange[0] + (valueRange[1] - valueRange[0]) * Math.random();
      if(dataType == "integer") {
        value = parseInt(value);
      }
      return value;
    }
    else {
      let startTime = valueRange[0].getTime();
      let endTime = valueRange[1].getTime();
      let value = startTime = (endTime - startTime) * Math.random();
      let date = new Date(value);
      let day = date.getDate(); //Date of the month: 2 in our example
      let month = date.getMonth() + 1; //Month of the Year: 0-based index, so 1 in our example
      let year = date.getFullYear() //Year: 2013
      return year + '-' + month + '-' + day;
    }
  }
  else {
    logExpression("It is an enumerable type.", 2);
    logExpression(valueRange, 2);
    return selectRandom(valueRange);
  }
}

function safeLowerCase(str) {
  if(!str) {
    return null;
  }
  let safe = str;
  try {
    safe = safe.toLowerCase();
  }
  catch(e) {
    logExpression("Got error in safeLowerCase.", 1);
    logExpression(e, 1);
    if(isNumeric(str)) {
      safe = safe.toString();
    }
  }
  return safe;
}

function isNumeric(x) {
  return !isNaN(parseFloat(x)) && !isNaN(x - 0);
}

function selectRandom(lst) {
  logExpression("In selectRandom with " + lst.length + " candidates.", 2);
  let indx = parseInt(Math.random() * lst.length);
  return lst[indx];
}

function generateRI(task) {
  logExpression("In generateRI with task: ", 3);
  logExpression(task, 3);

  let genRI = generateRandomInstantiation(task.abstract_graph, task.target_graph, task.instances, task.generate_values);
  genRI = genRI.map(el => {
    delete el.target_to_abstract;
    return el;
  })
  
  let output = {
    qNumber: task.qNumber,
    source_sql: task.source_sql,
    templatized_sql: task.templatized_sql,
    abstract_graph: task.abstract_graph,
    error: task.error,
    result: genRI
  }
  
  return output;
}

function transmogrifyGraph(abstractGraph, tableList) {
  let outEntities = [];
  let outRelationships = [];
  if(tableList) {
    abstractGraph.entities = abstractGraph.entities.filter(entity => {
      return tableList.includes(entity.type);
    });
  }

  abstractGraph.entities.forEach(tableBlock => {
    let tNode = {
      id: tableBlock.type.toLowerCase(),
      name: tableBlock.type.toLowerCase(),
      original_name: tableBlock.name,
      primaryKeys: tableBlock.id.split(','),
      description: tableBlock.description,
      type: "table"
    }

    outEntities.push(tNode);
    Object.keys(tableBlock.properties).forEach(column => {
      let props = tableBlock.properties[column];
      let cNode = {
        id: (tNode.id + '.' + column).toLowerCase(),
        name: column.toLowerCase(),
        description: props.description,
        type: "column",
        dataType: safeLowerCase(props.type)
      }
      if (props.max && props.min) {
        cNode.valueRange = [props.min, props.max];
      }
      if(props.values) {
        cNode.valueRange = props.values.filter(el => {
          return el;
        });
        if(!cNode.valueRange.length) {
          cNode.valueRange = null;
        }
      }

      outEntities.push(cNode);
      let relationship = {
        source: cNode.id.toLowerCase(),
        target: tNode.id.toLowerCase(),
        relationship: "parent"
      }
      outRelationships.push(relationship);
    });
  });
  let fkDict = {};
  abstractGraph.fk_relationships.forEach(relationship => {
    // Deal with possible case incompatibility, different fieldname (type vs relationship), and duplicate foreign keys
    let new_relationship = {
      source: relationship.source.toLowerCase(),
      target: relationship.target.toLowerCase(),
      relationship: relationship.type
    }
    let hash = new_relationship.source + ':' + new_relationship.target;
    fkDict[hash] = new_relationship;
  });
  Object.keys(fkDict).forEach(key => {
    outRelationships.push(fkDict[key]);
  });
  let outgraph = {entities: outEntities, relationships: outRelationships};
  return outgraph;
}

// Execute promises sequentially rather than in parallel to avoid errors resulting from excessive                                                
// parallel promises. Probably only needed by node.js, not Python.                                                                               
function iterateTask(blocks, fn) {
  return blocks.reduce((promiseChain, block) => {
    return promiseChain.then(chainResults =>
      fn(block).then(currentResult =>
        [...chainResults, currentResult]
      )
    );
  }, Promise.resolve([]))
}

function pruneTargetGraph(target_graph) {
  let badList = [];
  let newEntities = target_graph.entities.filter(node => {
    if(node.type == "column") {
      let vr = node.valueRange;
      let fail = !vr || !vr.length || (vr.length == 1 && !vr[0]);
      if(fail) {
        badList.push(node.id);
      }
      return !fail;
    }
    else {
      return true;
    }
  });
  logExpression("Finished filtering nodes.", 2);
  target_graph.entities = newEntities;

  let newRelationships = target_graph.relationships.filter(edge => {
    return !badList.includes(edge.source) && !badList.includes(edge.target);
  });
  logExpression("Finished filtering relationships.", 2);
  target_graph.relationships = newRelationships;
  logExpression("Returning target_graph with null valueRanges filtered out.", 2);

  return target_graph;
}

function addAutoForeignKeys(abstract_graph) {
  let dict = {};
  let entities = abstract_graph.entities;
  entities.forEach(eBlock => {
    let iArray = eBlock.id.split('.');
    if(eBlock.type == "column" && iArray.length == 2) {
      let column = iArray[1];
      if(!dict[column]) {
        dict[column] = [];
      }
      dict[column].push(eBlock.id);
    }
  });
  let fkRelationships = [];
  Object.keys(dict).forEach(column => {
    let lst = dict[column];
    for(i=0; i < lst.length - 1; i++) {
      for (j=i+1; j < lst.length; j++) {
        let source = lst[i];
        let target = lst[j];
        let rel = {
          source,
          target,
          relationship: "foreignKey"
        };
        fkRelationships.push(rel);
      }
    }
  });
  abstract_graph.relationships = abstract_graph.relationships.concat(fkRelationships);
  return abstract_graph;
}
