
import argparse
import logging
import os
import sys
import openai  # type: ignore
from typing import Optional

import pandas as pd
from get_prompt_template import get_prompt_template


# from dotenv import dotenv_values
from dotenv import load_dotenv  # type: ignore

from langchain_community.utilities.sql_database import SQLDatabase
from langchain_experimental.sql import SQLDatabaseChain
from langchain_ibm import WatsonxLLM
from langchain_openai import ChatOpenAI


#  Module global variables
__program_name__ = "create_paper2_pipeline1.py"
__version__ = 1.0
__updated__ = "2025-07-17"

logger = logging.getLogger(__program_name__)


def write_eval_files(
    src_db,
    src_gold_file,
    src_gold_sql,
    src_pred_file,
    src_pred_sql,
    target_db,
    target_gold_file,
    target_gold_sql,
    target_pred_file,
    target_pred_sql,
):

    if src_gold_file is not None:
        src_gold_file.write(f"{src_gold_sql}\t{src_db}\n")
        src_gold_file.flush()

    if src_pred_file is not None:
        src_pred_file.write(f"{src_pred_sql}\n")
        src_pred_file.flush()

    if target_gold_file is not None:
        target_gold_file.write(f"{target_gold_sql}\t{target_db}\n")
        target_gold_file.flush()

    if target_pred_file is not None:
        target_pred_file.write(f"{target_pred_sql}\n")
        target_pred_file.flush()


def get_query_df(queryfile):
    """Read queries from CSV file"""
    if queryfile.endswith(".csv"):
        df = pd.read_csv(queryfile)
        return df

    return None


def get_evalprefix(queryfile, model_type):

    # model_type = os.getenv("WATSONX_MODEL_ID")

    if model_type is None:
        raise ValueError("Env variable WATSONX_MODEL_ID is not defined}")

    # use an 'evalfiles' subdirectory for creating evalution input files
    queryfile_dir = os.path.dirname(queryfile)
    eval_dir = os.path.join(queryfile_dir, "evalfiles")

    os.makedirs(eval_dir, exist_ok=True)
    if "/" in model_type:
        model_name = model_type.split("/")[1]  # just get the last part for filenames
    else:
        model_name = model_type
    filename, _ = os.path.splitext(os.path.basename(queryfile))
    evalprefix = f"{filename}.{model_name}"

    # uncomment if we want to put all related eval files for a job in their own subdir
    # eval_dir = os.path.join(eval_dir, evalbase)
    evalbasename = os.path.join(eval_dir, evalprefix)

    return evalbasename


def create_llm_client_watsonx():

    WATSONX_PROJECT_ID = os.getenv("WATSONX_PROJECT_ID")
    WATSONX_APIKEY = os.getenv("WATSONX_APIKEY")
    WATSONX_URL = os.getenv("WATSONX_URL")
    WATSONX_MODEL_ID = os.getenv("WATSONX_MODEL_ID")

    logger.info("Using WatsonX LLM model '%s' for SQL generation", WATSONX_MODEL_ID)

    #  WatsonX LLM model parameters
    params = {
        "decoding_method": "greedy",
        # 'decoding_method': "sample",
        "max_new_tokens": 3000,
        "temperature": 0.05,
        "top_p": 0.1,
        "random_seed": 10,
        "repetition_penalty": 1.1,
        # 'time_limit': 5,
    }

    # create LLM
    llm = WatsonxLLM(
        model_id=WATSONX_MODEL_ID,
        url=WATSONX_URL,
        apikey=WATSONX_APIKEY,
        project_id=WATSONX_PROJECT_ID,
        params=params,
    )

    return llm


def create_llm_client_azure_openai():
    from langchain_openai import AzureChatOpenAI

    model_id = "gpt-4o-2024-08-06"
    print("------------------------envs:-----------")
    print(os.getenv("OPENAI_API_TYPE"))
    print(os.getenv("OPENAI_API_VERSION"))
    print(os.getenv("AZURE_OPENAI_API_KEY"))
    print(os.getenv("AZURE_OPENAI_ENDPOINT"))
    print(os.getenv("AZURE_OPENAI_API_VERSION"))
    llm = AzureChatOpenAI(
        deployment_name=model_id,
        temperature=0,
        seed=10,
    )

    logger.info(f"Using AZURE OpenAI LLM model {model_id} for SQL generation")
    return llm


def create_llm_client_HuggingFace(api_key=None, model_id=None):
    """
    Create a Hugging Face LLM model
    """
    # from langchain_community.llms import HuggingFaceHub
    # from langchain.llms import HuggingFaceHub
    # from langchain_community.llms import HuggingFaceEndpoint
    # from langchain_huggingface import HuggingFaceEndpoint
    # from langchain_huggingface.embeddings import HuggingFaceEndpointEmbeddings
    if api_key == None and model_id == None:
        api_key = os.getenv("HUGGINGFACE_API", None)
        os.environ["HUGGINGFACEHUB_API_TOKEN"] = api_key
        model_id = os.getenv("HuggingFace_MODEL_ID")

    # llm = HuggingFaceEndpoint(
    # llm = HuggingFaceEndpointEmbeddings(
    #     repo_id=model_id,
    #     # model_kwargs={
    #     #     "temperature": 0.05,
    #     #     "max_length": 3000,
    #     # },
    #     huggingfacehub_api_token=api_key,
    # )
    from langchain_huggingface import HuggingFacePipeline

    llm = HuggingFacePipeline.from_model_id(
        model_id=model_id,
        task="text-generation",
        pipeline_kwargs={
            "max_new_tokens": 2000,
            "top_k": 50,
            "temperature": 0.1,
        },
    )

    return llm


def nl2sql_watsonx(llm_client, question, target_schema):

    # TODO: Vey simple implmentation of NL2SQL just to get some output. Should be improved.
    messages = [
        {
            "role": "system",
            "content": "You are an SQL expert. Given a source natural language question, and a target schema, \
                your TASK is to generate an SQL statement which satisfies the question.",
        },
        {
            "role": "user",
            "content": "Here is the source question: "
            + question
            + "\nHere is the schema of the target domain: "
            + str(target_schema)
            + "\nGenerated target SQL: ",
        },
    ]

    response = llm_client.invoke(input=messages)
    logger.debug("LLM response '%s'", response)
    response = response.strip()

    # fix reponse which starts with a newline
    if response and response[0] == "\n":
        response = response[1:]

    # fix reponse which starts with a '|'
    if response and response[0] == "|":
        response = response[1:]

    # remove trailing extra output which some LLMs add to the response
    response = response.split("\n")[0]
    target_sql = response.strip()

    return target_sql


def create_llm_client_ccc(
    ccc_api_base, model, api_key="No Key!", rits: bool = False
):  # rits is a placeholder for future
    return ChatOpenAI(
        openai_api_key=api_key,
        openai_api_base=ccc_api_base,
        model=model,  # or your model name
    )


# SQLDatabaseChain with fix to strip code fences, used for azure openai
def strip_code_fence(sql: str) -> str:
    # Removes ```sql ... ``` or ``` ... ``` or leading/trailing backticks
    return (
        sql.strip()
        .removeprefix("```sql")
        .removeprefix("```")
        .removesuffix("```")
        .strip()
    )


# SQLDatabaseChain with fix to strip code fences, used for azure openai
def fix_openai_sql(sql: str) -> str:
    # Removes ```sql ... ``` or ``` ... ``` or leading/trailing backticks
    if sql is None or sql.strip() == "":
        return None
    index = sql.find("```sql")
    if index == -1:
        return None
    sql = sql[index:]
    return (
        sql.strip()
        .removeprefix("```sql")
        .strip()
        .removeprefix("SQLQuery:")
        .split("```", 1)[0]
        .strip()
        .replace("\n", " ")
        .strip()
    )


def fix_hf_sql(sql: str) -> str:
    """
    CREATE TABLE singer (
        "Singer_ID" INTEGER,
    ...
                Response format: SQL: SELECT * FROM asset where siteid = "Bedford"
                Replace the SQL output with your generated output.
    [/INST] SQL: SECTECT COUNT(*) FROM stadium WHERE Capacity > 10;


    ---
    [/INST] SQL:
    ```sql
    SQL: SELECT s.Name, s.Song_release_year
    FROM singer s ...
    """

    p = "SQL: "
    end = sql.rfind(";")
    if end == -1:
        end = sql.rfind("\n\n")

    if sql.find(p) == -1 or end == -1:
        return None
    sql = sql[sql.rfind(p) + len(p) : end+1]
    sql = sql.strip("```sql")
    sql = sql.strip().replace("\n", " ").strip()
    return sql


def process_queries(
    queryfile: str,
    databases: str,
    target_db_file: str,
    target_graph: str,
    startrow: Optional[int] = None,
    endrow: Optional[int] = None,
    rowindex: Optional[int] = None,
    use_ccc: bool = False,
    use_openai: bool = False,
    use_HuggingFace: bool = False,
    verbose: bool = False,
) -> bool:
    model = None
    if use_openai:
        model = "gpt-4o-2024-08-06"
        if os.getenv("AZURE_OPENAI_ENDPOINT") is None:
            logger.error("Env variable AZURE_OPENAI_ENDPOINT is not defined")
            return False

        if os.getenv("AZURE_OPENAI_API_KEY") is None:
            logger.error("Env variable AZURE_OPENAI_API_KEY is not defined")
            return False

        # create LLM client for WatsonX
        llm_client = create_llm_client_azure_openai()

    elif use_HuggingFace:

        model = os.getenv("HuggingFace_MODEL_ID")
        if model is None:
            logger.error("Env variable HuggingFace_MODEL_ID is not defined")
            return False

        if os.getenv("HUGGINGFACE_API") is None:
            logger.error("Env variable HUGGINGFACE_API is not defined")
            return False
        llm_client = create_llm_client_HuggingFace()

    elif use_ccc:
        # create LLM client for CCC
        ccc_api_base = os.getenv("CCC_API_BASE")
        if ccc_api_base is None:
            logger.error("Env variable CCC_API_BASE is not defined")
            return False

        ccc_api_key = os.getenv("CCC_API_KEY", "No Key!")
        if ccc_api_key == "No Key!":
            logger.warning(
                "Env variable CCC_API_KEY is not defined, using default value"
            )

        model = os.getenv("CCC_MODEL_NAME")

        llm_client = create_llm_client_ccc(ccc_api_base, model, ccc_api_key)
        logger.info(
            "Using CCC LLM model '%s' for SQL generation",
            os.getenv("CCC_MODEL_NAME", "Unknown"),
        )
        logger.info("Using CCC API base '%s'", ccc_api_base)
    else:
        # create LLM client for WatsonX

        model = os.getenv("WATSONX_MODEL_ID")
        if model is None:
            logger.error("Env variable WATSONX_MODEL_ID is not defined")
            return False

        if os.getenv("WATSONX_APIKEY") is None:
            logger.error("Env variable WATSONX_APIKEY is not defined")
            return False

        if os.getenv("WATSONX_URL") is None:
            logger.error("Env variable WATSONX_URL is not defined")
            return False

        if os.getenv("WATSONX_PROJECT_ID") is None:
            logger.error("Env variable WATSONX_PROJECT_ID is not defined")
            return False

        # create LLM client for WatsonX
        llm_client = create_llm_client_watsonx()

    if not os.path.isdir(databases):
        logger.error("Cannot find databases directory '%s'", databases)
        return False

    if target_graph is not None and not os.path.isfile(target_graph):
        logger.error("Cannot find target graph file '%s'", target_graph)
        return False

    if not os.path.isfile(target_db_file):
        logger.error("Cannot find target database file '%s'", target_db_file)
        return False

    # extract target db name and parent directory
    filestem, _ = os.path.splitext(target_db_file)
    paths = str.split(filestem, os.path.sep)
    target_db_name = paths[-1]
    db_dir = paths[-2]

    if target_db_name != db_dir:
        logger.error(
            "Target database filename '%s' does not match containing directory name '%s'",
            target_db_name,
            db_dir,
        )
        return False

    target_db_uri = f"sqlite:///{target_db_file}"

    if target_graph is None:
        target_db = SQLDatabase.from_uri(target_db_uri, sample_rows_in_table_info=0)
    else:
        from redacted_libraryname.agent.tools import ExtendedSQLDatabase

        target_db = ExtendedSQLDatabase.from_uri(
            target_db_uri, sample_rows_in_table_info=0
        )
        target_db.data_dir = os.path.join(*paths[0:-2])
        target_db.abstract_graph_file = target_graph

    if use_openai:
        # Patch the database runner to fix SQL before execution
        original_run = target_db.run
        target_db.run = lambda query: original_run(strip_code_fence(query))

    target_db_chain = SQLDatabaseChain.from_llm(
        llm_client, target_db, verbose=verbose, return_sql=True, top_k=100000
    )

    logger.info("Using src DB directory '%s'", databases)
    logger.info("Reading query file '%s'", queryfile)
    query_df = get_query_df(queryfile)
    if query_df is None:
        logger.error("Could not create dataframe from query file '%s'", queryfile)
        return False

    for col in (
        "SourceDB",
        "SourceQuestion",
        "SourceSQL",
        "TargetQuestion",
        "TargetSQL",
    ):
        if col not in query_df.columns:
            logger.error("Column '%s' not found in query file '%s'", col, queryfile)
            return False

    evalprefix = get_evalprefix(queryfile, model)

    # sql query files needed for running final evaluation
    src_gold_file = f"{evalprefix}.src.gold.txt"
    src_pred_file = f"{evalprefix}.src.predict.txt"

    target_gold_file = f"{evalprefix}.target.gold.txt"
    target_pred_file = f"{evalprefix}.target.predict.txt"

    with open(src_gold_file, "wt", encoding="utf8") as src_gold, open(
        src_pred_file, "wt", encoding="utf8"
    ) as src_pred, open(target_gold_file, "wt", encoding="utf8") as target_gold, open(
        target_pred_file, "wt", encoding="utf8"
    ) as target_pred:

        skip_count = 0
        for i, row in query_df.iterrows():
            index = i + 1
            if startrow is not None and index < startrow:
                continue
            if endrow is not None and index > endrow:
                break

            # only handle the specified query
            if rowindex is not None and rowindex < index:
                break
            if rowindex is not None and rowindex != index:
                continue

            src_db_name = row["SourceDB"]
            src_question = row["SourceQuestion"]
            src_gold_sql = row["SourceSQL"]

            target_question = row["TargetQuestion"]
            target_gold_sql = row["TargetSQL"]

            logger.info("==========================================================")
            logger.info(
                "Processing question number '%i', from '%s' database",
                index,
                src_db_name,
            )
            logger.info("Source Question(%d): '%s'", index, src_question)
            logger.info("Source Golden SQL(%d): '%s'", index, src_gold_sql)
            logger.info("Target Question(%d): '%s'", index, target_question)
            logger.info("Target Golden SQL(%d): '%s'", index, target_gold_sql)

            # some Target queries are not executable, filter these out for now using
            # the 'NumRows' column if available
            if "NumRows" in row:
                if row["NumRows"] == 0:
                    logger.warning(
                        "Target Golden SQL(%d) produces 0 rows, skipping", index
                    )
                    skip_count += 1
                    continue
            else:
                try:
                    results = target_db.run(target_gold_sql)
                    if len(results) == 0:
                        logger.warning(
                            "Target Golden SQL(%d) produces 0 rows, skipping...", index
                        )
                        skip_count += 1
                        continue
                except Exception:
                    logger.warning(
                        "Target Golden SQL(%d) is not executable, skipping", index
                    )
                    skip_count += 1
                    continue

            src_db_file = os.path.join(databases, src_db_name, src_db_name + ".sqlite")
            if not os.path.isfile(src_db_file):
                logger.error("Cannot find source DB file '%s'", src_db_file)
                return False

            src_db_uri = f"sqlite:///{src_db_file}"
            logger.debug("src_db_uri: '%s'", src_db_uri)
            src_db = SQLDatabase.from_uri(src_db_uri, sample_rows_in_table_info=0)

            if use_openai:
                # Patch the database runner to fix SQL before execution
                original_run = src_db.run
                src_db.run = lambda query: original_run(strip_code_fence(query))

            if not use_HuggingFace:  # we do not use langchain for HF
                src_db_chain = SQLDatabaseChain.from_llm(
                    llm_client, src_db, verbose=verbose, return_sql=True, top_k=100000
                )

            try:
                src_generated_sql = None
                target_generated_sql = None
                if not use_HuggingFace:
                    src_generated_sql = src_db_chain.run(src_question)
                else:  # use_HuggingFace
                    schema = src_db.get_table_info()
                    prompt_template = get_prompt_template(model)
                    formatted_prompt = prompt_template.format(
                        dialect="sqlite",
                        utterance=src_question,
                        schema=schema,
                    )
                    src_generated_sql = llm_client.invoke(formatted_prompt)

                logger.info(
                    "-----src_generated_sql:%s--------------", src_generated_sql
                )

                if use_openai:
                    src_generated_sql = fix_openai_sql(src_generated_sql)

                elif use_HuggingFace:
                    src_generated_sql = fix_hf_sql(src_generated_sql)

                else:
                    src_generated_sql = src_generated_sql.splitlines()[0]

                if not src_generated_sql or src_generated_sql.strip() == "":
                    logger.warning(
                        "Failed to create Generated SQL for question '%d', skipping...",
                        index,
                    )
                    write_eval_files(
                        src_db=src_db_name,
                        src_gold_file=src_gold,
                        src_gold_sql=src_gold_sql,
                        src_pred_file=src_pred,
                        src_pred_sql="FAILURE",
                        target_db=target_db_name,
                        target_gold_file=target_gold,
                        target_gold_sql=target_gold_sql,
                        target_pred_file=target_pred,
                        target_pred_sql="FAILURE",
                    )
                    continue

                logger.info(
                    "Source Generated SQL(%d): '%s'",
                    index,
                    str(src_generated_sql).strip(),
                )

                target_generated_sql = None
                if not use_HuggingFace:
                    target_generated_sql = target_db_chain.run(target_question)
                else:
                    schema = target_db.get_table_info()
                    prompt_template = get_prompt_template(model)
                    formatted_prompt = prompt_template.format(
                        dialect="sqlite",
                        utterance=target_question,
                        schema=schema,
                    )
                    target_generated_sql = llm_client.invoke(formatted_prompt)

                logger.info(
                    "-----target_generated_sql:%s--------------", target_generated_sql
                )
                if use_openai:
                    target_generated_sql = fix_openai_sql(target_generated_sql)
                elif use_HuggingFace:
                    target_generated_sql = fix_hf_sql(target_generated_sql)
                else:
                    target_generated_sql = target_generated_sql.splitlines()[0]
                if not target_generated_sql or target_generated_sql.strip() == "":
                    logger.warning(
                        "Failed to create Generated SQL for question '%d', skipping...",
                        index,
                    )
                    write_eval_files(
                        src_db=src_db_name,
                        src_gold_file=src_gold,
                        src_gold_sql=src_gold_sql,
                        src_pred_file=src_pred,
                        src_pred_sql=src_generated_sql,
                        target_db=target_db_name,
                        target_gold_file=target_gold,
                        target_gold_sql=target_gold_sql,
                        target_pred_file=target_pred,
                        target_pred_sql="FAILURE",
                    )
                    continue

                logger.info(
                    "Target Generated SQL(%d): '%s'",
                    index,
                    str(target_generated_sql).strip(),
                )
                write_eval_files(
                    src_db=src_db_name,
                    src_gold_file=src_gold,
                    src_gold_sql=src_gold_sql,
                    src_pred_file=src_pred,
                    src_pred_sql=src_generated_sql,
                    target_db=target_db_name,
                    target_gold_file=target_gold,
                    target_gold_sql=target_gold_sql,
                    target_pred_file=target_pred,
                    target_pred_sql=target_generated_sql,
                )

            except Exception as ex:
                logger.error("Failed processing question(%d): '%s'", index, str(ex))
                src_generated_sql = (
                    src_generated_sql if bool(src_generated_sql) else "FAILURE"
                )
                target_generated_sql = (
                    target_generated_sql if bool(target_generated_sql) else "FAILURE"
                )

                write_eval_files(
                    src_db=src_db_name,
                    src_gold_file=src_gold,
                    src_gold_sql=src_gold_sql,
                    src_pred_file=src_pred,
                    src_pred_sql=src_generated_sql,
                    target_db=target_db_name,
                    target_gold_file=target_gold,
                    target_gold_sql=target_gold_sql,
                    target_pred_file=target_pred,
                    target_pred_sql=target_generated_sql,
                )
                continue

        if skip_count:
            logger.info("Skipped '%d' 0 row queries", skip_count)

    return True


def parse_args(argv=None):
    """Command line options."""
    program_name = __program_name__
    program_version = __version__
    program_build_date = __updated__
    program_eplilog = program_name + f"({program_version}, built {program_build_date})"

    program_desc = "Create Paper2 Pipeline-1 evaluation data"

    if argv is None:
        argv = sys.argv[1:]

    # setup option parser
    parser = argparse.ArgumentParser(
        prog=program_name, description=program_desc, epilog=program_eplilog
    )

    # Inputs
    parser.add_argument(
        "--env",
        "-e",
        type=str,
        default=None,
        help="Env file to load settings/credentials, default(.env)",
    )

    parser.add_argument(
        "--query-file",
        "-q",
        dest="queryfile",
        type=str,
        required=True,
        help="Input queries CSV file containing SourceSQL, SourceQuestion, TargetSQL and TargetQuestion columns",
    )

    parser.add_argument(
        "--src-db-dir",
        "--db",
        type=str,
        dest="databases",
        required=True,
        help="Databases directory containing source sqlite databases e.g. data/source/spider_data/databases",
    )

    parser.add_argument(
        "--target-db-file",
        type=str,
        dest="target_db_file",
        required=True,
        help="Target sqlite database file e.g. data/target/asset/asset.sqlite",
    )

    parser.add_argument(
        "--target-graph",
        type=str,
        dest="target_graph",
        required=False,
        default=None,
        help="abstract graph json file for target domain, e.g. abstract_graph.json",
    )

    parser.add_argument(
        "--start",
        type=int,
        dest="startrow",
        default=None,
        help="Only process queries starting from the specifed row, (default: %(default)s)",
    )

    parser.add_argument(
        "--end",
        type=int,
        dest="endrow",
        default=None,
        help="Last query index to evaluate, (default: %(default)s)",
    )

    parser.add_argument(
        "--row",
        type=int,
        dest="row",
        default=None,
        help="Only process query at the given row index, (default: %(default)s)",
    )

    parser.add_argument(
        "--fileprefix",
        action="store_true",
        help="Print output filename prefix and quit, used for shell scripting, (default: %(default)s)",
    )

    # Settings
    parser.add_argument(
        "-v",
        "--verbose",
        action="store_true",
        help="Enable more verbose log messages (default: %(default)s)",
    )

    group1 = parser.add_mutually_exclusive_group()
    group1.add_argument(
        "--use_ccc",
        action="store_true",
        dest="use_ccc",
        help="Use LLM from model deployed on CCC",
    )
    group1.add_argument(
        "--use_openai",
        action="store_true",
        dest="use_openai",
        help="Use openai LLM",
    )
    group1.add_argument(
        "--use_HuggingFace",
        action="store_true",
        dest="use_HuggingFace",
        help="Use HuggingFace LLM",
    )

    args = parser.parse_args()

    return args


def main(argv=None):
    """

    Args:
        argv (_type_, optional): arguments. Defaults to None.
    """

    args = parse_args(argv)

    logging.basicConfig(
        level=logging.INFO, format="%(levelname)s: %(name)s:%(lineno)d: %(message)s"
    )

    logger.info(
        "Loading environment file '%s'", args.env if args.env is not None else ".env"
    )
    # config = dotenv_values(args.env)

    load_dotenv(args.env, override=True)

    # special case:
    # # just echo the eval filename prefix, used by a calling shell script
    if args.fileprefix:
        if args.use_ccc:
            model = os.getenv(
                "CCC_MODEL_NAME",
                "20250530-G3codeinstruct3b-ITAutmation50k-RP125k-checkpoint-65778",
            )
        elif args.use_openai:
            model = "gpt-4o-2024-08-06"  # TODO get from env

        elif args.use_HuggingFace:
            model = os.getenv("HuggingFace_MODEL_ID")
        else:
            model = os.getenv("WATSONX_MODEL_ID", "granite_3.1-8b_withsql")

        evalbase = get_evalprefix(args.queryfile, model)
        print(evalbase)  # this print is required for the --fileprefix option to work
        return 0

    # suppress INFO messages for some very chatty packages
    # pylint: disable=E1101
    loggers = [logging.getLogger(name) for name in logging.root.manager.loggerDict]
    for lgr in loggers:
        if lgr.name in (
            "httpx",
        ):
            # print(lgr.name, lgr.level)
            lgr.setLevel(logging.ERROR)

    process_queries(
        queryfile=args.queryfile,
        databases=args.databases,
        target_db_file=args.target_db_file,
        target_graph=args.target_graph,
        startrow=args.startrow,
        endrow=args.endrow,
        rowindex=args.row,
        use_ccc=args.use_ccc,
        use_openai=args.use_openai,
        use_HuggingFace=args.use_HuggingFace,
        verbose=args.verbose,
    )


if __name__ == "__main__":
    sys.exit(main())
