#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
"""

import os
import numpy as np


def fast_scandir_for_folders(folder:str)->list:
    r"""
    Fast scan of folder for subfolders to be searched for the data
    Used for shape data
    """
    
    subfolders= [f.path for f in os.scandir(folder) if f.is_dir()]
    for dir in list(subfolders):
        subfolders.extend(fast_scandir_for_folders(dir))
    
    return subfolders if subfolders else []+[folder] #return the current folder if subfolders is empty

def load_shape_data(folders:list):
    """ 
    Load shape data
    
    Parameters
    ----------
    folders : list of folders
    
    Returns
    -------

    data: array [n, ] of n arrays i of dimensions (m_i, 3)
    label: array [n,] of labels
    
    """
    data = []
    labels = []
    for subfold in folders:
        for f in os.listdir(subfold):
            if 'vert' in f:
                data.append(np.loadtxt(subfold+'/'+f))
                l = ''.join([c.lower() for c in f.split('.')[0] if c.isalpha()])
                labels.append(l)

    return data, labels


def readOffFile(filename:str):
    """Reads an OFF README.md and returns the vertices and triangles in numpy arrays.

    The OFF README.md is read and the data captured in the array structures.
    This is a fairly trivial reading task.

    Args:
        | filename (string): name of the OFF README.md

    Returns:
        | vertices(np.array([])): array of vertices as [x y z]
        | triangles(np.array([])): array of triangles as []

    Raises:
        | No exception is raised.
    Credits : courtesy of https://www.programcreek.com/python/?CodeExample=read+off
   """
    with open(filename) as f:
        #first line [0] has only the word OFF
        lines = f.readlines()
        if lines[0].find('OFF') < 0:
            print('not an OFF README.md')
            return (None,None)
        
        #second line [1] has counts for ....
        counts = lines[1].split()
        vertexCount = int(counts[0])
        faceCount = int(counts[1])
        
        # then follows vertices from lines[2] to lines[2+vertexCount]
        vertices = np.asarray([float(s) for s in lines[2].split()])
        for line in lines[3:2+vertexCount]:
            vertices = np.vstack((vertices,np.asarray([float(s) for s in line.split()])))
            
        # now extract the triangles lines[2+vertexCount] to lines(-1)
        triangles = np.asarray([int(s) for s in lines[2+vertexCount].split()[1:]],dtype='i4')
        for line in lines[3+vertexCount:2+vertexCount+faceCount]:
            if len(line) > 0:
                triangles = np.vstack((triangles,np.asarray([int(s) \
                    for s in line.split()[1:]])))

        if triangles.shape[0] != faceCount or vertices.shape[0] != vertexCount:
            return (None,None)
        else:
            return (vertices, triangles)


def load_off_format_data(folder:str):
    """ 
    Load shape data
    
    Parameters
    ----------
    folder : folder containing off format files
    
    Returns
    -------

    data: array [n, ] of n arrays i of dimensions (m_i, 3)
    label: array [n,] of labels
    transform : array[n,] of applied transformations
    st: array[n,] of the transform strenght
    
    """
    data, labels, transform, st = [], [], [], []
    for f in os.listdir(folder):
        x, _ = readOffFile(folder+'/'+f)
        data.append(x)
        aux = f.split('.')
        labels.append(int(aux[0]))
        transform.append(aux[1])
        st.append(int(aux[2]))
        
    return (np.asarray(data, dtype='object'), np.asarray(labels), np.asarray(transform), np.asarray(st))


def write_to_txt(dist_mat, n_points, filename):

    with open(filename,'wt') as f:
        f.write(''.join(str(n_points))+'\n')
        [f.write(' '.join([str(it) for it in item])+'\n') for item in dist_mat]
        
        
def samples_n_vertices(X, n:int):
    
    for i in range(X.shape[0]):
        ind = np.random.choice(X[i].shape[0], n, replace=False)
        X[i] = X[i][ind]

    return X