from struct import pack, unpack
from collections import defaultdict
import numpy as np

"""
                  IO Utils
"""


def read_fvecs(filename, max_size=None):
    max_size = max_size or float('inf')
    with open(filename, "rb") as f:
        vecs = []
        while True:
            header = f.read(4)
            if not len(header): break
            dim, = unpack('<i', header)
            vec = unpack('f' * dim, f.read(4 * dim))
            vecs.append(vec)
            if len(vecs) >= max_size: break
    return np.array(vecs, dtype="float32")


def read_dvecs(filename, max_size=None):
    max_size = max_size or float('inf')
    with open(filename, "rb") as f:
        vecs = []
        while True:
            header = f.read(4)
            if not len(header): break
            dim, = unpack('<i', header)
            vec = unpack('d' * dim, f.read(8 * dim))
            vecs.append(vec)
            if len(vecs) >= max_size: break
    return np.array(vecs, dtype="float64")


def read_ivecs(filename, max_size=None):
    max_size = max_size or float('inf')
    with open(filename, "rb") as f:
        vecs = []
        while True:
            header = f.read(4)
            if not len(header): break
            dim, = unpack('<i', header)
            vec = unpack('i' * dim, f.read(4 * dim))
            vecs.append(vec)
            if len(vecs) >= max_size: break
    return np.array(vecs, dtype="int32")



def write_fvecs(filename, vecs):
    with open(filename, "wb") as f:
        for vec in vecs:
            dim = len(vec)
            f.write(pack('<i', dim))
            f.write(pack('f' * dim, *list(vec)))

            
def write_dvecs(filename, vecs):
    with open(filename, "wb") as f:
        for vec in vecs:
            dim = len(vec)
            f.write(pack('<i', dim))
            f.write(pack('d' * dim, *list(vec)))
            

def write_ivecs(filename, vecs):
    with open(filename, "wb") as f:
        for vec in vecs:
            dim = len(vec)
            f.write(pack('<i', dim))
            f.write(pack('i' * dim, *list(vec)))
            
            
def write_fdata(filename, vecs):
    with open(filename, "wb") as f:
        for vec in vecs:
            dim = len(vec)
            f.write(pack('f' * dim, *list(vec)))
            
            
def write_idata(filename, vecs):
    with open(filename, "wb") as f:
        for vec in vecs:
            dim = len(vec)
            f.write(pack('i' * dim, *list(vec)))