import os
import re
from datetime import timedelta
from pathlib import Path

import torch
import torch.distributed as dist
import torch.multiprocessing as mp
from configs import create_workshop, get_config
from engine import Engine


def main_worker(local_rank, cfg, mode, world_size, dist_url, i):
    torch.cuda.empty_cache()
    torch.cuda.set_device(local_rank)
    asr.utils.environment.set_seed(cfg.train.seed)
    """ To enable deterministic behavior in this case, you must set an environment variable before running 
    your PyTorch application: CUBLAS_WORKSPACE_CONFIG=:4096:8 or CUBLAS_WORKSPACE_CONFIG=:16:8. """
    os.environ["CUBLAS_WORKSPACE_CONFIG"] = ":4096:8"

    dist.init_process_group(
        backend="nccl",
        init_method=dist_url,
        timeout=timedelta(seconds=6000),
        world_size=world_size,
        rank=local_rank,
    )

    if cfg.model.init_with_ckpt:
        mark = re.search("(?<=_mark_)\w+", cfg.model.path_to_vesper)
        if mark is not None:
            if cfg.mark is None:
                cfg.mark = mark.group()
            else:
                cfg.mark = mark.group() + "_" + cfg.mark

    cfg = create_workshop(cfg, mode, local_rank)
    cfg.ckpt_save_path = str(Path(cfg.train.resume).parent)
    print(f"cfg.ckpt_save_pat: {cfg.ckpt_save_path}")
    # shutil.copy(cfg.train.resume, cfg.ckpt_save_path)
    engine = Engine(cfg, mode, local_rank, world_size)
    engine.prepare_staff()
    engine.evaluate(save_model=False, ith_layer_inference=i)
    for j, dataloader in enumerate(engine.dataloader_test):
        engine.test(dataloader, split_name=engine.cfg.dataset.test_folder[j], ith_layer_inference=i)


def main(cfg, mode, i):
    asr.utils.environment.set_seed(cfg.train.seed)
    free_port = asr.utils.distributed.find_free_port()
    dist_url = f"tcp://127.0.0.1:{free_port}"
    world_size = torch.cuda.device_count()  # num_gpus
    print(f"world_size={world_size} Using dist_url={dist_url}")
    mp.spawn(fn=main_worker, args=(cfg, mode, world_size, dist_url, i), nprocs=world_size)


if __name__ == "__main__":
    mode = "_finetune"
    cfg = get_config(mode=mode)
    for i in range(24):
        main(cfg, mode, i)
