import matplotlib.pyplot as plt
import h5py
import numpy as np

import skfem as fem
from skfem.helpers import dot, grad  # helpers make forms look nice
from skfem.element import ElementQuad1
from scipy.interpolate import RegularGridInterpolator
from skfem import MeshQuad



path='/path/to/directory/'#darcy_flow.h5'

with h5py.File(path+'darcy_flow.h5') as f:
    data = f['tensor'][:]
    nu = f['nu'][:]

nnodes=128
nx, ny = 2*(nnodes,) 
x = np.linspace(0, 1, nx)
y = np.linspace(0, 1, ny)



Nx, Ny = nnodes, nnodes  # number of square elements in x and y directions
mesh = MeshQuad().init_tensor(
    x=np.linspace(0, 1, Nx),
    y=np.linspace(0, 1, Ny))


element = ElementQuad1()
Vh = fem.Basis(mesh, element)


@fem.BilinearForm
def diffusion(u, v, _):
    return dot(grad(u), grad(v))


A_dif= diffusion.assemble(Vh)

@fem.LinearForm
def l(v, w):
    x, y = w.x  
    f = 20 * np.sin(np.pi * x) * np.sin(np.pi * y)
    return f * v


b = l.assemble(Vh)
D = Vh.get_dofs()

u_list=[]
input_list=[]

samples=10000

for i in range(samples):
    vel=nu[i]
    
    VX=np.where(vel==1, 5*vel, vel)
    VY=np.where(vel==1, 5*vel, vel)

    speed = np.hypot(VX, VY)

    vx_interp = RegularGridInterpolator((x, y), VX)
    vy_interp = RegularGridInterpolator((x, y), VY)

    @fem.BilinearForm
    def advection(u, v, w):
        v_field=np.stack([vx_interp(w.x.T).T,vy_interp(w.x.T).T],axis=0)
        return u*dot(grad(v),v_field)


    A_adv= advection.assemble(Vh)
    u = fem.solve(*fem.condense(A_dif+A_adv, b, D=D))

    input=np.reshape(VX,(nnodes,-1))
    
    input_list.append(np.reshape(VX,(nnodes,-1)))
    u_list.append(np.reshape(u,(nnodes,-1)))

    if (i%50==0):
        print(f"Step : {i}/{samples}")

u_list=np.stack(u_list)
input_list=np.stack(input_list)

with h5py.File(path+'darcy_advection.h5', 'w') as f:
    f.create_dataset('vel', data=input_list)
    f.create_dataset('u', data=u_list)
