"""Module for generation of Qualtrics surveys."""

import argparse
import json
import logging
import random
from copy import deepcopy
from pathlib import Path

import pandas as pd
from rich.console import Console
from rich.logging import RichHandler
from rich.table import Table

logging.basicConfig(
    level='INFO',
    format='%(message)s',
    datefmt='[%X]',
    handlers=[RichHandler()],
)


class QualtricsSurvey:
    """A class for creating a (basic) Qualtrics survey."""

    def __init__(
        self,
    ):
        """Initialize the Qualtrics survey."""
        self.qid = 1
        self.save_directory = Path('survey')

        with Path('survey', 'survey_template').with_suffix('.json').open() as f:
            self.survey = json.load(f)

        with Path('survey', 'question_template').with_suffix('.json').open() as f:
            self.question_template = json.load(f)

        with Path('survey', 'block_template').with_suffix('.json').open() as f:
            self.blocks = json.load(f)

    def format_question_as_html(
        self,
        question: str,
        explanation_a: str,
        explanation_b: str,
    ) -> str:
        """Format a question as HTML.

        Parameters
        ----------
        question: str
            The question to format.

        Returns
        -------
        str
            The formatted question.

        """
        html_question = f"""
            <!-- Question prompt -->
            <div
                style="text-align:center; font-size:1.2em; font-weight:bold; font-style:italic; margin: 1em 0; padding:0.5em; border-bottom:2px solid #ccc;">
                {question}
            </div>

            <!-- Explanations side-by-side -->
            <table style="width:100%; table-layout:fixed; border-collapse:collapse; margin-top:1em;">
                <tr>
                    <td style="vertical-align:top;text-align:left;border-right:2px solid #ccc;padding:2em; width:50%;">
                        <div style="text-align:center; font-weight:bold; margin-bottom:1em;">Explanation A</div>
                        {explanation_a}
                    </td>
                    <td style="vertical-align:top;text-align:left;padding:2em; width:50%;">
                        <div style="text-align:center; font-weight:bold; margin-bottom:1em;">Explanation B</div>
                        {explanation_b}
                    </td>
                </tr>
            </table>

            <!-- Footer with annotation guidelines -->
            <div style="text-align:left; font-size:1em; margin: 1em 0; padding:0.5em; border-top:2px solid #ccc;">
                <strong>Reminder:</strong> when answering the questions below, do not consider minor formatting points, only
                consider the information given in the explanation.
            </div>
        """

        return html_question

    def add_question_to_survey(
        self,
        uuid: str,
        question: str,
    ):
        """Add a question to a Qualtrics survey.

        Parameters
        ----------
        uuid: str
            The UUID of the question/model/dataset combination.
        question: dict
            The question from a given dataset.

        """
        new_question = deepcopy(self.question_template)
        new_question['PrimaryAttribute'] = f'QID{self.qid}'
        new_question['SecondaryAttribute'] = uuid
        new_question['Payload']['QuestionText'] = question
        new_question['Payload']['DataExportTag'] = uuid
        new_question['Payload']['QuestionDescription'] = uuid
        new_question['Payload']['QuestionID'] = f'QID{self.qid}'

        self.survey['SurveyElements'].append(new_question)

    def add_question_to_block(
        self,
        block_id: str,
    ):
        """Add a question to a block of a Qualtrics survey.

        Parameters
        ----------
        block_id: str
            The block ID.

        """
        try:
            block_index = next(index for index, block in enumerate(self.blocks['Payload']) if block['Description'] == block_id)
        except StopIteration:
            raise ValueError(f'Block "{block_id}" not found.') from StopIteration

        # Create the new block element
        new_element = {
            'Type': 'Question',
            'QuestionID': f'QID{self.qid}',
        }

        # Append the new element to the block's elements
        self.blocks['Payload'][block_index]['BlockElements'].append(new_element)

        # Add QID to random subset if not an attention check
        if block_id not in ['introduction', 'primer', 'guidelines', 'attention-check-a', 'attention-check-no-pref']:
            self.blocks['Payload'][block_index]['Options']['Randomization']['Advanced']['RandomSubSet'].append(f'QID{self.qid}')

    def fill(
        self,
        questions: pd.DataFrame | str,
    ) -> pd.DataFrame:
        """Fill the survey with questions.

        Parameters
        ----------
        questions: pd.DataFrame or str
            The questions DataFrame, or path to survey.jsonl file.

        Returns
        -------
        pd.DataFrame
            DataFrame of questions added.

        """
        rows = []
        with Path(questions).open(encoding='utf-8') as f:
            lines = [line for line in f if line.strip()]

        logging.info(f'Filling survey with {len(lines)} questions from {questions}...')
        n = len(lines)
        split = n // 2

        for i, line in enumerate(lines):
            entry = json.loads(line)
            id = entry.get('id')
            question = entry.get('question', '')
            base_expl = entry.get('base_explanation', '')
            with_report_expl = entry.get('with_report_explanation', '')

            # Randomise explanation positions
            if random.random() < 0.5:
                explanation_a = base_expl
                explanation_b = with_report_expl
                explanation_a_entry = 'base'
                explanation_b_entry = 'with_report'
            else:
                explanation_a = with_report_expl
                explanation_b = base_expl
                explanation_a_entry = 'with_report'
                explanation_b_entry = 'base'

            html = self.format_question_as_html(
                question=question,
                explanation_a=explanation_a,
                explanation_b=explanation_b,
            )

            self.add_question_to_survey(id, html)
            # Assign to block-1 or block-2 by index
            block_id = 'block-1' if i < split else 'block-2'
            self.add_question_to_block(block_id)
            self.qid += 1

            rows.append(
                {
                    'qid': self.qid,
                    'uuid': id,
                    'html': html,
                    'question': question,
                    'base_explanation': base_expl,
                    'with_report_explanation': with_report_expl,
                    'explanation_a': explanation_a_entry,
                    'explanation_b': explanation_b_entry,
                }
            )

        # Now add attention check questions (html is stored in .txt files)
        attention_checks = ['attention-check-a', 'attention-check-no-pref']
        for check in attention_checks:
            with open(Path('survey', f'{check}.txt'), encoding='utf-8') as f:
                html = f.read()

            self.add_question_to_survey(check, html)
            self.add_question_to_block(check)
            self.qid += 1

        # Add introduction
        with open(Path('survey', 'introduction.txt'), encoding='utf-8') as f:
            intro_html = f.read()
        self.add_question_to_survey('introduction', intro_html)
        self.add_question_to_block('introduction')
        self.qid += 1

        # Add primer
        with open(Path('survey', 'primer.txt'), encoding='utf-8') as f:
            primer_html = f.read()
        self.add_question_to_survey('primer', primer_html)
        self.add_question_to_block('primer')
        self.qid += 1

        # Add guidelines
        with open(Path('survey', 'guidelines.txt'), encoding='utf-8') as f:
            guidelines_html = f.read()
        self.add_question_to_survey('guidelines', guidelines_html)
        self.add_question_to_block('guidelines')
        self.qid += 1

        # Add the blocks element to the survey
        self.survey['SurveyElements'].append(self.blocks)
        return pd.DataFrame(rows)

    def save_survey(
        self,
        questions: pd.DataFrame,
        save: str = 'filled_survey_for_upload',
    ) -> pd.DataFrame:
        """Save the survey to a JSON file.

        Parameters
        ----------
        save: str, optional
            The filename to save the survey.

        """
        logging.info(f'Saving survey to {self.save_directory} as {save}.json and {save}.qsf')
        with Path(self.save_directory, save).with_suffix('.json').open('w') as f:
            json.dump(self.survey, f, indent=4)

        with Path(self.save_directory, save).with_suffix('.qsf').open('w') as f:
            f.write(json.dumps(self.survey))

        logging.info(f'Saving survey content to {self.save_directory} as {save}.jsonl')
        questions.to_json(Path(self.save_directory, save).with_suffix('.jsonl'), orient='records', lines=True)

        # Preview survey content by printing questions table using rich
        console = Console()
        table = Table(show_header=True, header_style='bold magenta')
        table.add_column('ID', style='dim')
        table.add_column('Question')
        table.add_column('Base Explanation')
        table.add_column('With Report Explanation')

        for _, row in questions.head().iterrows():
            table.add_row(
                str(row['qid']),
                row['question'],
                row['base_explanation'],
                row['with_report_explanation'],
            )

        # console.print(table)

        return questions


if __name__ == '__main__':
    parser = argparse.ArgumentParser(description='Prepare inputs for a Qualtrics survey.')
    parser.add_argument(
        '--source-material', type=str, default='graphs/explanations/survey.jsonl', help='Path to source material for questions.'
    )
    args = parser.parse_args()

    Q = QualtricsSurvey()
    questions_df = Q.fill(questions=args.source_material)
    Q.save_survey(questions_df)
