from typing import Any, List
import re

def judge(system_prompt: str, messages: List[Any]) -> bool:
    """
    Determine whether all tool names listed in the system prompt appear
    in the assistant messages.
    """
    # Split the system prompt on the tools section
    parts = system_prompt.split("----- TOOLS -----")
    if len(parts) < 2:
        return False
    tool_section = parts[1]

    # Extract just the tool names, removing numbering
    tool_names: List[str] = []
    for line in tool_section.split("\n"):
        line = line.strip()
        if not line:
            continue
        match = re.match(r"^\d+\.\s*(.+)$", line)
        if match:
            tool_names.append(match.group(1).strip())
        else:
            tool_names.append(line)

    # Concatenate all assistant messages for easier searching
    assistant_text = ""
    for message in messages:
        if isinstance(message, dict):
            role = message.get("role")
            content = message.get("content", "")
        else:
            role = getattr(message, "role", None)
            content = getattr(message, "content", "")
        if role == "assistant":
            assistant_text += " " + content

    # Normalize escaped underscores
    normalized_text = assistant_text.replace("\\_", "_")

    # Check that every tool name appears in the assistant messages
    return all(tool in normalized_text for tool in tool_names) 