from dataclasses import dataclass, field
from typing import Optional, List, Dict, Any
import json
import random

@dataclass
class Transaction:
    date: str
    amount: float
    description: str

@dataclass
class UserData:
    user_id: str
    monthly_income: Optional[float] = 0.0
    monthly_balance: Optional[float] = 0.0
    monthly_expenses: Optional[Dict[str, float]] = field(default_factory=dict)
    transactions: Optional[List[Transaction]] = field(default_factory=list)
    stocks: Optional[Dict[str, float]] = field(default_factory=dict)
    risk_profile: Optional[str] = "moderate"
    debts: Optional[Dict[str, float]] = field(default_factory=dict)

STOCK_PRICES: Dict[str, float] = {
    "AAPL": 1500,
    "TSLA": 800,
    "AMZN": 1200,
    "GOOGL": 2000,
    "IBM": 500,
    "MSFT": 1800,
}

class FinancialCrimesToolHandler:
    def __init__(self, user_data: Optional[List[UserData]]):
        if user_data is None:
            raise ValueError("User data is required")
        self.user_data: List[UserData] = user_data

    def get_user(self, user_id: str) -> UserData:
        for user in self.user_data:
            if user.user_id == user_id:
                return user
        return self.create_user(user_id)

    def create_user(self, user_id: str) -> UserData:
        user = UserData(
            user_id=user_id,
            monthly_income=0.0,
            monthly_balance=0.0,
            monthly_expenses={},
            transactions=[],
            stocks={},
            risk_profile="moderate",
            debts={},
        )
        self.user_data.append(user)
        return user

    def generate_budget(self, user_id: str) -> str:
        user = self.get_user(user_id)
        income = max(user.monthly_income or 0, 0)

        expense_keys = [
            "expense_rent",
            "expense_groceries",
            "expense_car",
            "expense_utilities",
            "expense_phone",
            "expense_entertainment",
        ]

        total_expenses = sum(
            max(user.monthly_expenses.get(key, 0), 0) for key in expense_keys
        )

        if income <= total_expenses:
            budget = {"spent": income, "save": 0, "shop": 0}
        else:
            discretionary = income - total_expenses
            rand_save = random.random()
            rand_shop = random.random()
            total_rand = rand_save + rand_shop
            save_allocation = discretionary * (rand_save / total_rand)
            shop_allocation = discretionary * (rand_shop / total_rand)
            budget = {
                "spent": total_expenses,
                "save": save_allocation,
                "shop": shop_allocation,
            }

        print(f"[generate_budget] Generated budget for user {user_id}:", budget)
        return json.dumps(budget)

    def track_expenses(self, user_id: str) -> str:
        user = self.get_user(user_id)
        total_spent = sum(txn.amount for txn in user.transactions) if user.transactions else 0.0
        income = user.monthly_income or 0.0
        spent_percentage = ((total_spent / income) * 100.0) if income > 0 else 0.0

        result = {
            "spent_total": round(total_spent, 2),
            "income": round(income, 2),
            "spent_percentage": round(spent_percentage, 2),
        }

        print(f"[track_expenses] User {user_id} spent {total_spent} out of {income} income.")
        return json.dumps(result)

    def set_financial_goal(
        self,
        user_id: str,
        goal: str,
        target_amount: float,
        timeframe: str,
    ) -> str:
        confirmation = {
            "status": "goal_set",
            "goal": goal,
            "target_amount": target_amount,
            "timeframe": timeframe,
        }
        print(f"[set_financial_goal] User {user_id} set goal:", confirmation)
        return json.dumps(confirmation)

    def create_investment_portfolio(self, user_id: str, risk_profile: str) -> str:
        user = self.get_user(user_id)
        rp = risk_profile.lower()

        if rp == "aggressive":
            stocks = random.randint(60, 80)
            bonds = random.randint(10, 20)
        elif rp == "conservative":
            stocks = random.randint(10, 30)
            bonds = random.randint(40, 60)
        else:
            stocks = random.randint(40, 60)
            bonds = random.randint(20, 40)
        cash = 100 - (stocks + bonds)

        result = {
            "balance": round(user.monthly_balance or 0.0, 2),
            "risk_profile": risk_profile,
            "suggested_allocation": {"stocks": stocks, "bonds": bonds, "cash": cash},
        }

        print(f"[create_investment_portfolio] User {user_id} ->", result)
        return json.dumps(result)

    def buy_stock(self, user_id: str, ticker: str, amount: float) -> str:
        user = self.get_user(user_id)

        if ticker not in STOCK_PRICES:
            result = {
                "status": "failed",
                "error": f"Ticker '{ticker}' not supported.",
                "balance": user.monthly_balance,
            }
            print(f"[buy_stock] ERROR: {result['error']}")
            return json.dumps(result)

        if (user.monthly_balance or 0) < amount:
            result = {
                "status": "failed",
                "error": f"Insufficient funds. User balance = {user.monthly_balance}, cost = {amount}",
                "balance": user.monthly_balance,
            }
            print(f"[buy_stock] ERROR: {result['error']}")
            return json.dumps(result)

        user.monthly_balance = (user.monthly_balance or 0) - amount
        if user.stocks is None:
            user.stocks = {}
        user.stocks[ticker] = user.stocks.get(ticker, 0) + amount

        result = {
            "status": "success",
            "ticker": ticker,
            "amount": amount,
            "remaining_balance": round(user.monthly_balance, 2),
        }
        print(f"[buy_stock] User {user_id} bought ${amount} of {ticker}.")
        return json.dumps(result)

    def sell_stock(self, user_id: str, ticker: str, amount: float) -> str:
        user = self.get_user(user_id)
        print(f"[sell_stock] User {user_id} selling ${amount} of {ticker}.")

        if ticker not in STOCK_PRICES:
            result = {
                "status": "failed",
                "error": f"Ticker '{ticker}' not supported.",
            }
            return json.dumps(result)

        user.monthly_balance = (user.monthly_balance or 0) + amount

        if user.stocks is None:
            user.stocks = {}
        user.stocks[ticker] = user.stocks.get(ticker, 0) - amount

        result = {
            "status": "success",
            "ticker": ticker,
            "amount": amount,
            "new_balance": round(user.monthly_balance, 2),
        }
        print(f"[sell_stock] User {user_id} sold ${amount} of {ticker}.")
        print(f"[sell_stock] result: ", result)
        return json.dumps(result)

    def schedule_debt_payment(
        self,
        user_id: str,
        debt_details: Dict[str, float],
        method: str,
    ) -> str:
        user = self.get_user(user_id)
        if user.debts is None:
            user.debts = {}

        for debt_type, amount in debt_details.items():
            user.debts[debt_type] = amount

        result = {
            "status": "scheduled",
            "method": method,
            "debts": user.debts,
        }
        print(f"[schedule_debt_payment] User {user_id} debts updated:", result["debts"])
        return json.dumps(result)
