from dataclasses import dataclass, field, asdict
from typing import List, Optional, Dict, Any
import json
import random

@dataclass
class FinancialSummary:
    annual_income: float
    credit_score: float
    pre_approved_mortgage_amount: float

@dataclass
class DesiredLocation:
    city: str
    state: str

@dataclass
class AccountInfo:
    bank_name: str
    account_number: str
    balance: float

@dataclass
class Property:
    location: str
    price: float
    bedrooms: int
    bathrooms: int
    features: List[str]
    property_id: str
    owner_user_id: Optional[str] = None

@dataclass
class UserData:
    name: str
    address: str
    email: str
    phone_number: str
    user_id: str
    financial_summary: FinancialSummary
    preferred_property_types: List[str]
    desired_locations: List[DesiredLocation]
    must_have_features: List[str]
    move_in_timeframe: str
    current_listings: List[Property]
    account_info: AccountInfo

# We'll populate this with properties from users' current_listings
property_data: Dict[str, Property] = {}

class ListingPriceToolHandler:
    def __init__(self, user_data: List[UserData]):
        if not user_data:
            raise ValueError("User data is required")
        self.user_data = user_data

        # Initialize property_data with all properties from all users
        self.populate_property_data()

    # New method to populate property_data from user listings
    def populate_property_data(self):
        # Clear existing data
        property_data.clear()

        # Add properties from all users
        for user in self.user_data:
            if user.current_listings and isinstance(user.current_listings, list):
                for listing in user.current_listings:
                    # Add owner_user_id to track ownership
                    property_with_owner = Property(
                        **{**asdict(listing), "owner_user_id": user.user_id}
                    )
                    property_data[listing.property_id] = property_with_owner

        print(
            f"[populatePropertyData] Loaded {len(property_data)} properties"
        )

    def get_user(self, user_id: str) -> UserData:
        user = next((u for u in self.user_data if u.user_id == user_id), None)
        return user or self.create_user(user_id)

    def create_user(self, user_id: str) -> UserData:
        user = UserData(
            name="",
            address="",
            email="",
            phone_number="",
            user_id=user_id,
            financial_summary=FinancialSummary(
                annual_income=0,
                credit_score=0,
                pre_approved_mortgage_amount=0,
            ),
            preferred_property_types=[],
            desired_locations=[],
            must_have_features=[],
            move_in_timeframe="",
            current_listings=[],
            account_info=AccountInfo(
                bank_name="",
                account_number="",
                balance=0,
            ),
        )
        self.user_data.append(user)
        return user

    def search_properties(
        self,
        user_id: str,
        criteria: Dict[str, Any],
    ) -> str:
        user = self.get_user(user_id)

        location = criteria.get("location", "")
        location = location.lower() if location else ""
        price_range_str = criteria.get("price_range", "")
        min_price = 0
        max_price = float("inf")

        if "-" in price_range_str:
            min_str, max_str = price_range_str.split("-", 1)
            try:
                min_price = int(min_str) if min_str else 0
            except Exception:
                min_price = 0
            try:
                max_price = int(max_str) if max_str else float("inf")
            except Exception:
                max_price = float("inf")

        min_bed = criteria.get("bedrooms", 0)
        min_bath = criteria.get("bathrooms", 0)
        features = criteria.get("features", [])
        if isinstance(features, list):
            desired_features = features
        elif features:
            desired_features = [features]
        else:
            desired_features = []

        results = []
        for prop in property_data.values():
            if location and location not in prop.location.lower():
                continue
            if not (min_price <= prop.price <= max_price):
                continue
            if prop.bedrooms < min_bed or prop.bathrooms < min_bath:
                continue
            if any(
                not any(f.lower() == feat.lower() for f in prop.features)
                for feat in desired_features
            ):
                continue
            results.append(asdict(prop))

        result = {
            "status": "success",
            "user_id": user_id,
            "matching_properties": results,
        }
        print(
            f"[search_properties] Found {len(results)} matching properties for user {user.name}."
        )
        return json.dumps(result)

    def analyze_market(self, area: str) -> str:
        sample_market_insights = [
            f"Average home price in {area} is up 5% from last year.",
            f"Inventory in {area} is relatively low, leading to competitive offers.",
            f"Average days on market in {area} has dropped to 21 days.",
        ]
        chosen_insight = random.choice(sample_market_insights)

        result = {
            "status": "analysis_complete",
            "area": area,
            "insight": chosen_insight,
        }
        print(
            f"[analyze_market] Market analysis for {area}: {chosen_insight}"
        )
        return json.dumps(result)

    def schedule_tour(self, user_id: str, property_id: str, date: str) -> str:
        user = self.get_user(user_id)
        if property_id not in property_data:
            result = {
                "status": "failed",
                "error": f"Property {property_id} not found.",
            }
            print(f"[schedule_tour] ERROR: {result['error']}")
            return json.dumps(result)

        result = {
            "status": "tour_scheduled",
            "user_id": user_id,
            "property_id": property_id,
            "scheduled_date": date,
        }
        print(
            f"[schedule_tour] Tour scheduled for {user.name} at property {property_id} on {date}."
        )
        return json.dumps(result)

    def provide_mortgage_info(
        self,
        user_id: str,
        financial_details: Dict[str, Any],
    ) -> str:
        user = self.get_user(user_id)
        possible_rates = [3.5, 4.0, 4.25, 4.5]
        chosen_rate = random.choice(possible_rates)

        down_payment = financial_details.get("down_payment", 0)
        approx_monthly_payment = round(
            ((200000 - down_payment) * (chosen_rate / 100)) / 12, 2
        )

        result = {
            "status": "mortgage_info_generated",
            "user_id": user_id,
            "annual_income": financial_details.get("annual_income", 0),
            "credit_score": financial_details.get("credit_score", 0),
            "down_payment": down_payment,
            "sample_interest_rate": chosen_rate,
            "approx_monthly_payment": approx_monthly_payment,
        }
        print(
            f"[provide_mortgage_info] Provided mortgage info for {user.name}. Rate: {chosen_rate}%"
        )
        return json.dumps(result)

    def process_transaction(
        self,
        user_id: str,
        transaction_details: Dict[str, Any],
    ) -> str:
        user = self.get_user(user_id)
        balance = user.account_info.balance
        amount = transaction_details.get("amount", 0)

        if amount > balance:
            result = {
                "status": "failed",
                "error": "Insufficient funds",
            }
            print(
                f"[process_transaction] ERROR: {result['error']} for {user.name}."
            )
            return json.dumps(result)

        user.account_info.balance = balance - amount
        result = {
            "status": "transaction_success",
            "user_id": user_id,
            "property_id": transaction_details.get("property_id"),
            "amount": amount,
            "from_account": transaction_details.get("from_account"),
            "to_account": transaction_details.get("to_account"),
            "remaining_balance": user.account_info.balance,
        }
        print(
            f"[process_transaction] Transaction successful for {user.name}. Amount: {amount}"
        )
        return json.dumps(result)

    def guide_buying_process(self, user_id: str, property_id: str) -> str:
        user = self.get_user(user_id)
        steps = [
            "1. Finalize your budget and mortgage pre-approval.",
            "2. Schedule property tours and inspections.",
            "3. Negotiate the offer and finalize contract.",
            "4. Complete all necessary paperwork and closing.",
        ]

        result = {
            "status": "guidance_provided",
            "user_id": user_id,
            "property_id": property_id,
            "steps": steps,
        }
        print(
            f"[guide_buying_process] Provided buying process steps to {user.name}."
        )
        return json.dumps(result)

    def transfer_funds(
        self,
        user_id: str,
        from_account: str,
        to_account: str,
        amount: float,
    ) -> str:
        user = self.get_user(user_id)
        balance = user.account_info.balance

        if amount > balance:
            result = {
                "status": "failed",
                "error": "Insufficient funds to transfer",
            }
            print(f"[transfer_funds] ERROR: {result['error']} for {user.name}.")
            return json.dumps(result)

        user.account_info.balance = balance - amount
        result = {
            "status": "transfer_success",
            "user_id": user_id,
            "from_account": from_account,
            "to_account": to_account,
            "amount": amount,
            "remaining_balance": user.account_info.balance,
        }
        print(
            f"[transfer_funds] {amount} transferred from {from_account} to {to_account} for {user.name}."
        )
        return json.dumps(result)

    def verify_property_ownership(self, user_id: str, property_id: str) -> str:
        user = self.get_user(user_id)
        # Check both the user's listings and our property_data map
        is_owner = (
            any(listing.property_id == property_id for listing in user.current_listings)
            or (property_id in property_data and property_data[property_id].owner_user_id == user_id)
        )

        result = {
            "status": "ownership_verified" if is_owner else "ownership_denied",
            "user_id": user_id,
            "property_id": property_id,
            "is_owner": is_owner,
        }
        print(
            f"[verify_property_ownership] Ownership check: {user.name}, {property_id}, {is_owner}"
        )
        return json.dumps(result)

    def update_property_listing(
        self,
        property_id: str,
        listing_updates: Dict[str, Any],
    ) -> str:
        if property_id not in property_data:
            result = {
                "status": "failed",
                "error": f"Property {property_id} not found.",
            }
            print(f"[update_property_listing] ERROR: {result['error']}")
            return json.dumps(result)

        # Get the property from our data store
        property_obj = property_data[property_id]

        # Apply updates
        for k, v in listing_updates.items():
            if hasattr(property_obj, k):
                setattr(property_obj, k, v)

        # Also update in the user's listings if they own it
        owner = next(
            (user for user in self.user_data if user.user_id == property_obj.owner_user_id),
            None,
        )
        if owner:
            user_listing = next(
                (listing for listing in owner.current_listings if listing.property_id == property_id),
                None,
            )
            if user_listing:
                for k, v in listing_updates.items():
                    if hasattr(user_listing, k):
                        setattr(user_listing, k, v)

        result = {
            "status": "update_success",
            "property_id": property_id,
            "updated_fields": listing_updates,
        }
        print(
            f"[update_property_listing] Updated {property_id} with {listing_updates}"
        )
        return json.dumps(result)
