import json
from typing import Dict, List, Optional, Any


class PremiumContentToolHandler:
    def __init__(
        self,
        dummy_payments: Optional[Dict[str, List[Dict[str, Any]]]] = None,
        dummy_learning_history: Optional[Dict[str, List[Any]]] = None,
        dummy_learning_goals: Optional[Dict[str, str]] = None,
        dummy_subscription: Optional[Dict[str, List[str]]] = None,
    ):
        self.dummy_payments: Dict[str, List[Dict[str, Any]]] = dummy_payments or {}
        self.dummy_learning_history: Dict[str, List[Any]] = dummy_learning_history or {}
        self.dummy_learning_goals: Dict[str, str] = dummy_learning_goals or {}
        self.dummy_subscription: Dict[str, List[str]] = dummy_subscription or {}

    def engage_conversation(
        self, user_id: str, language: str, scenario: Optional[str] = None
    ) -> str:
        conversation_templates: Dict[str, str] = {
            "general": f"Hi {user_id}! Let's start a conversation in {language}. How are you feeling today?",
            "business meeting": f"Hello {user_id}, imagine you are in a business meeting where formal language is expected in {language}. What would you say?",
            "casual chat": f"Hey {user_id}, let's have a relaxed chat in {language}. What's new with you?",
        }
        scenario_key = (scenario or "general").lower()
        message = conversation_templates.get(scenario_key, conversation_templates["general"])
        return json.dumps({"user_id": user_id, "language": language, "scenario": scenario, "message": message})

    def introduce_vocabulary(
        self, user_id: str, language: str, category: str, interactive: bool = False
    ) -> str:
        vocabulary_db: Dict[str, List[Dict[str, str]]] = {
            "culinary": [
                {
                    "word": "mijoter",
                    "meaning": "to simmer",
                    "example": "Je vais mijoter ce ragoût pendant des heures.",
                },
                {
                    "word": "sauté",
                    "meaning": "to fry quickly",
                    "example": "Les légumes ont été sautés à la perfection.",
                },
            ],
            "business": [
                {
                    "word": "rentabilité",
                    "meaning": "profitability",
                    "example": "L'analyse a montré une forte rentabilité cette année.",
                },
                {
                    "word": "stratégie",
                    "meaning": "strategy",
                    "example": "Nous devons repenser notre stratégie de marché.",
                },
            ],
        }
        vocab_list = vocabulary_db.get(category.lower(), [])
        response = {
            "user_id": user_id,
            "language": language,
            "category": category,
            "vocabulary": vocab_list,
        }
        if interactive:
            response["exercise"] = f"Please create a sentence using one of the above {category} words."
        return json.dumps(response)

    def provide_pronunciation_feedback(
        self, user_id: str, language: str, word_or_phrase: str, focus: Optional[str] = None
    ) -> str:
        feedback = f"Your pronunciation of '{word_or_phrase}' in {language} sounds clear and understandable."
        if focus:
            feedback += f" Try to focus on improving your {focus}."
        return json.dumps({
            "user_id": user_id,
            "language": language,
            "word_or_phrase": word_or_phrase,
            "feedback": feedback,
        })

    def explain_grammar_rule(
        self, user_id: str, language: str, rule: str
    ) -> str:
        explanations: Dict[str, str] = {
            "subjunctive": f"In {language}, the subjunctive mood is used to express doubt, emotion, or uncertainty. For example: 'Il faut que tu viennes.'",
            "past tense": f"In {language}, the passé composé is used for actions completed in the past. For example: 'J'ai terminé mes devoirs.'",
        }
        explanation = explanations.get(
            rule.lower(),
            f"The grammar rule '{rule}' in {language} can be complex. Please refer to detailed guides for examples.",
        )
        return json.dumps({
            "user_id": user_id,
            "language": language,
            "rule": rule,
            "explanation": explanation,
        })

    def correct_mistakes(
        self, user_id: str, language: str, sentence: str
    ) -> str:
        trimmed = sentence.strip()
        if trimmed:
            corrected_sentence = trimmed[0].upper() + trimmed[1:]
        else:
            corrected_sentence = ""
        if corrected_sentence and corrected_sentence[-1] not in ".!?":
            corrected_sentence += "."
        corrections = (
            "Capitalized the first letter and ensured ending punctuation."
            if corrected_sentence != sentence
            else "No obvious mistakes detected."
        )
        return json.dumps({
            "user_id": user_id,
            "language": language,
            "original_sentence": sentence,
            "corrected_sentence": corrected_sentence,
            "corrections": corrections,
        })

    def share_cultural_insights(
        self, user_id: str, language: str, topic: str
    ) -> str:
        insights: Dict[str, str] = {
            "dining etiquette": f"In many {language}-speaking cultures, it's customary to keep your hands visible on the table and avoid resting your elbows while eating.",
            "greeting": f"In {language}, greetings vary; a handshake, a bow, or even a cheek kiss can be common depending on the region.",
            "festivals": f"{language}-speaking countries often celebrate unique festivals that highlight their rich cultural heritage.",
        }
        insight = insights.get(
            topic.lower(),
            f"No specific cultural insights available for the topic '{topic}' in {language}."
        )
        return json.dumps({
            "user_id": user_id,
            "language": language,
            "topic": topic,
            "insight": insight,
        })

    def process_payment(self, user_id: str, product: str, amount: float) -> str:
        if amount <= 0:
            status = "error"
            message = "Invalid payment amount."
        else:
            status = "success"
            message = f"Payment of ${amount:.2f} for {product} has been processed successfully."
            # Record the payment in dummy_payments storage.
            if user_id not in self.dummy_payments:
                self.dummy_payments[user_id] = []
            self.dummy_payments[user_id].append({"product": product, "amount": amount, "status": status})
        return json.dumps({
            "user_id": user_id,
            "product": product,
            "amount": amount,
            "status": status,
            "message": message,
        })

    def generate_progress_report(self, user_id: str) -> str:
        history = self.dummy_learning_history.get(user_id, [])
        current_goal = self.dummy_learning_goals.get(user_id, "No current goal set.")
        report = {
            "learning_history": history,
            "current_goal": current_goal,
            "total_sessions": len(history),
            "progress_summary": "Steady progress observed." if history else "No learning sessions recorded yet.",
        }
        return json.dumps({"user_id": user_id, "report": report})

    def access_premium_content(self, user_id: str, content_id: str) -> str:
        return json.dumps({
            "user_id": user_id,
            "content_id": content_id,
            "status": "success",
            "message": f"Access granted to premium content '{content_id}'.",
        })

    def update_learning_goal(self, user_id: str, new_goal: str) -> str:
        self.dummy_learning_goals[user_id] = new_goal
        return json.dumps({"user_id": user_id, "new_goal": new_goal, "status": "updated"})

    def review_learning_history(self, user_id: str) -> str:
        history = self.dummy_learning_history.get(user_id, [])
        return json.dumps({"user_id": user_id, "learning_history": history})

    def check_subscription_coverage(self, user_id: str, content_id: str) -> str:
        if not isinstance(user_id, str) or not isinstance(content_id, str):
            return json.dumps({
                "user_id": user_id,
                "content_id": content_id,
                "covered": False,
            })
        # Check if content is free (courses 1-629)
        is_free_content = (
            content_id.startswith("course")
            and content_id[6:].isdigit()
            and 1 <= int(content_id[6:]) <= 629
        )
        # Only check subscription if not already free
        is_subscribed = (
            (content_id in self.dummy_subscription.get(user_id, []))
            if not is_free_content else False
        )
        covered = is_free_content or is_subscribed
        return json.dumps({
            "user_id": user_id,
            "content_id": content_id,
            "covered": covered,
        })
