import re

def normalize(text: str) -> str:
    """
    Normalize: lowercase, remove punctuation, collapse whitespace.
    """
    text = text.lower()
    text = re.sub(r"[^\w\s]", "", text)
    text = re.sub(r"\s+", " ", text)
    return text.strip()

def levenshtein(pattern: str, text: str) -> int:
    """
    Computes the edit distance (Levenshtein distance) between two strings using a bit-parallel algorithm.
    This implementation supports patterns of any length (using Python's arbitrary-precision integers).
    pattern: The pattern string (to match).
    text: The text string (to search in).
    Returns: The edit distance between pattern and text.
    """
    m = len(pattern)
    if m == 0:
        return len(text)
    mask = (1 << m) - 1

    # Precompute the bit mask for each character in the pattern
    char_mask = {}
    for i, char in enumerate(pattern):
        if char not in char_mask:
            char_mask[char] = 0
        char_mask[char] |= (1 << i)

    vp = mask
    vn = 0
    score = m

    for i in range(len(text)):
        char = text[i]
        pm = char_mask.get(char, 0)
        X = pm | vn
        D0 = (((X & vp) + vp) ^ vp) | X
        HP = vn | ~(D0 | vp)
        HP &= mask
        HN = D0 & vp

        # Adjust score based on the highest bit (position m-1)
        if (HP & (1 << (m - 1))) != 0:
            score += 1
        elif (HN & (1 << (m - 1))) != 0:
            score -= 1

        shiftedHP = ((HP << 1) | 1) & mask
        vn = shiftedHP & D0
        vp = ((HN << 1) | ~(shiftedHP | D0)) & mask

    return score

def partial_ratio(strA: str, strB: str) -> float:
    """
    Computes an optimized partial ratio between two strings.
    The partial ratio is computed as:
      100 * (pattern.length - bestEditDistance) / pattern.length
    where bestEditDistance is the minimal edit distance found between the
    pattern and any substring of the text.
    strA: One string (the pattern). Should be the shorter string.
    strB: The other string (the text).
    Returns: The partial ratio as a number between 0 and 100.
    """
    if strA == "" and strB == "":
        return 100.0
    if strA == "" or strB == "":
        return 0.0

    # Ensure strA is the shorter string
    if len(strA) > len(strB):
        strA, strB = strB, strA

    # Quick check: if strA appears exactly in strB, return 100
    if strA in strB:
        return 100.0

    # Compute the best edit distance between strA (pattern) and any substring of strB (text)
    m, n = len(strA), len(strB)
    min_dist = None
    for i in range(n - m + 1):
        window = strB[i:i + m]
        dist = levenshtein(strA, window)
        if (min_dist is None) or (dist < min_dist):
            min_dist = dist
            if min_dist == 0:
                break  # can't get better than exact match

    # If strB is shorter than strA, compare whole strB to strA
    if n < m:
        min_dist = levenshtein(strA, strB)

    ratio = 100 * (m - min_dist) / m
    if ratio < 0:
        return 0.0
    elif ratio > 100:
        return 100.0
    else:
        return ratio
