from dataclasses import dataclass, field, asdict
from typing import List, Optional, Dict, Any
import json
import random

@dataclass
class LeaveBalanceData:
    vacation: int = 0
    sick: int = 0

@dataclass
class Employee:
    user_id: str
    name: str = ""
    position: str = ""
    department: str = ""
    email: str = ""
    phone_number: str = ""
    manager_name: str = ""
    manager_id: str = ""
    leave_balance: Dict[str, int] = field(default_factory=lambda: {"vacation": 0, "sick": 0})
    current_trainings: List[str] = field(default_factory=list)
    career_goals: List[str] = field(default_factory=list)

@dataclass
class InterviewSchedule:
    status: str
    requester_id: str
    candidate_id: str
    interviewer_id: str
    date: str

@dataclass
class TrainingRecommendation:
    employee: str
    career_goal: str
    recommended_trainings: List[str]

@dataclass
class LeaveBalance:
    status: Optional[str] = None
    error: Optional[str] = None
    employee: str = ""
    leave_type: str = ""
    current_balance: int = 0

@dataclass
class LeaveRequest:
    status: str
    error: Optional[str] = None
    employee: Optional[str] = None
    leave_type: Optional[str] = None
    requested_days: Optional[int] = None
    available_balance: Optional[int] = None
    remaining_balance: Optional[int] = None

class WithoutApprovalToolHandler:
    def __init__(self, employeeData: List[Employee]):
        self.employeeData: List[Employee] = employeeData

    def getEmployee(self, employeeId: str) -> Employee:
        for u in self.employeeData:
            if u.user_id == employeeId:
                return u
        return self.createEmployee(employeeId)

    def createEmployee(self, employeeId: str) -> Employee:
        employee = Employee(
            user_id=employeeId,
            name="",
            position="",
            department="",
            email="",
            phone_number="",
            manager_name="",
            manager_id="",
            leave_balance={"vacation": 0, "sick": 0},
            current_trainings=[],
            career_goals=[],
        )
        self.employeeData.append(employee)
        return employee

    def scheduleInterview(
        self,
        requesterId: str,
        candidateId: str,
        interviewerId: str,
        date: str,
        _managerApproval: bool = False,
        _candidateConfirmation: bool = False,
    ) -> str:
        result = InterviewSchedule(
            status="interview_confirmed",
            requester_id=requesterId,
            candidate_id=candidateId,
            interviewer_id=interviewerId,
            date=date,
        )
        print(f"[schedule_interview] {json.dumps(asdict(result))}")
        return json.dumps(asdict(result))

    def processLeaveRequest(
        self,
        employeeId: str,
        leaveType: str,
        duration: int,
    ) -> str:
        employee = self.getEmployee(employeeId)
        leaveTypeLower = leaveType.lower()

        if leaveTypeLower not in employee.leave_balance:
            result = LeaveRequest(
                status="failed",
                error=f"Invalid leave type '{leaveType}'.",
            )
            print(f"[process_leave_request] ERROR: {result.error}")
            return json.dumps(asdict(result))

        currentBalance = employee.leave_balance[leaveTypeLower]
        if duration > currentBalance:
            result = LeaveRequest(
                status="denied",
                employee=employee.name,
                leave_type=leaveType,
                requested_days=duration,
                available_balance=currentBalance,
            )
            print(f"[process_leave_request] WARNING: Not enough balance for {leaveType}.")
            return json.dumps(asdict(result))

        # Deduct from balance
        employee.leave_balance[leaveTypeLower] = currentBalance - duration
        updatedBalance = employee.leave_balance[leaveTypeLower]
        result = LeaveRequest(
            status="approved",
            employee=employee.name,
            leave_type=leaveType,
            requested_days=duration,
            remaining_balance=updatedBalance,
        )
        print(f"[process_leave_request] Leave approved for {employeeId}.")
        return json.dumps(asdict(result))

    def recommendTraining(self, employeeId: str, careerGoal: str) -> str:
        employee = self.getEmployee(employeeId)

        sampleTrainings = [
            "Leadership Workshop",
            "Advanced Data Analytics",
            "Effective Communication",
            "Cloud Architecture Mastery",
            "UI/UX In Depth",
            "Project Management Professional (PMP) Prep",
        ]

        suggested = random.sample(sampleTrainings, 2)

        result = TrainingRecommendation(
            employee=employee.name,
            career_goal=careerGoal,
            recommended_trainings=suggested,
        )

        print(f"[recommend_training] For {employeeId} with goal '{careerGoal}', recommended: {suggested}")
        return json.dumps(asdict(result))

    def trackLeaveBalance(
        self,
        employeeId: str,
        _requesterId: str,
        leaveType: str,
    ) -> str:
        employee = self.getEmployee(employeeId)
        leaveTypeLower = leaveType.lower()

        if leaveTypeLower not in employee.leave_balance:
            result = LeaveBalance(
                status="failed",
                error=f"Invalid leave type '{leaveType}'.",
                employee="",
                leave_type="",
                current_balance=0,
            )
            print(f"[track_leave_balance] ERROR: {result.error}")
            return json.dumps(asdict(result))

        currentBalance = employee.leave_balance[leaveTypeLower]
        result = LeaveBalance(
            employee=employee.name,
            leave_type=leaveType,
            current_balance=currentBalance,
        )
        print(f"[track_leave_balance] {leaveType} balance for {employee.name}.")
        return json.dumps(asdict(result))
