from typing import Any, Dict, List, Optional

def requires_inline_tool_results(model: Optional[str]) -> bool:
    """
    Checks if the model requires inline tool results instead of separate tool messages.
    """
    import re
    return bool(model and re.match(r"^(amazon)", model))

def requires_mistral_ordering(model: Optional[str]) -> bool:
    """
    Checks if the model requires Mistral ordering instead of OpenAI ordering.
    """
    import re
    return bool(model and re.match(r"^(mistral)", model))

def is_google_model(model: Optional[str]) -> bool:
    """
    Checks if the model is a Google model that can only handle one tool call at a time.
    """
    import re
    return bool(model and re.match(r"^google", model))

def create_inline_tool_results_content(
    tool_responses: List[Optional[Dict[str, Any]]]
) -> str:
    """
    Creates the inline tool results content string.
    """
    return "".join(
        f'<toolResult id="{tr.get("tool_call_id", "")}">{tr.get("content", "")}</toolResult>'
        for tr in tool_responses if tr
    )

def apply_inline_tool_results_to_openai_message(
    message: Dict[str, Any],
    tool_results_string: str,
) -> None:
    """
    Applies tool results to an OpenAI-compatible message.
    """
    if isinstance(message.get("content"), str):
        message["content"] = f'{message["content"]} {tool_results_string}'
    elif isinstance(message.get("content"), list):
        message["content"].append({
            "type": "text",
            "text": tool_results_string,
        })

def execute_tool_for_id(
    tool_call: Dict[str, Any],
    tools: List[Any],
    message_context: Optional[Any] = None,
    existing_messages: Optional[List[Any]] = None,
) -> Optional[Dict[str, Any]]:
    """
    Executes a tool call and gets its response.
    """
    tool = next((t for t in tools if t["function"]["name"] == tool_call["function"]["name"]), None)
    if not tool:
        return None

    # Skip if this tool call ID has already been used in messages
    if (
        existing_messages
        and any(m.get("tool_call_id") == tool_call["id"] for m in existing_messages)
    ):
        return None

    import json
    args = json.loads(tool_call["function"].get("arguments") or "{}")
    # The tool's execute method may be a function or method
    if hasattr(tool, "execute"):
        # tool_response = tool.execute(args) if message_context is None else tool.execute.__call__(message_context, args)
        tool_response = tool.execute(args)
    else:
        # tool_response = tool["execute"](args) if message_context is None else tool["execute"].__call__(message_context, args)
        tool_response = tool["execute"](args)

    return {
        "role": "tool",
        "tool_call_id": tool_call["id"],
        "content": tool_response,
    }

def transform_messages(
    messages: List[Dict[str, Any]],
    bucket: Any,
    model_type: str,
    tools: Optional[List[Any]] = None,
    message_context: Optional[Any] = None,
    model: Optional[str] = None,
) -> List[Dict[str, Any]]:
    """
    Transforms messages for the target model, handling attachments, tool calls, and ordering.
    """
    transformed_messages: List[Dict[str, Any]] = []

    for message in messages:
        content = message.get("content")

        # Handle attachments (images)
        if isinstance(message.get("attachments"), list) and message["attachments"]:
            content = [{"type": "text", "text": message["content"]}]

        message_base = {
            "role": message["role"],
            "content": content,
        }

        extra_mistral_message_arr: List[Dict[str, Any]] = []

        openai_extras = {}
        if model_type == "OpenAI":
            tool_calls = message.get("tool_calls")
            openai_extras = {
                "tool_call_id": message.get("tool_call_id") or (tool_calls[0]["id"] if tool_calls and len(tool_calls) > 0 else None),
                "tool_calls": [
                    {
                        **tool_call,
                        "function": {
                            **tool_call["function"],
                            "arguments": (
                                tool_call["function"]["arguments"]
                                if isinstance(tool_call["function"]["arguments"], str)
                                else (
                                    str(tool_call["function"]["arguments"])
                                    if tool_call["function"]["arguments"] is not None
                                    else "{}"
                                )
                            ) or "{}"
                        }
                    }
                    for tool_call in tool_calls
                ] if isinstance(tool_calls, list) and tool_calls else None,
            }

        # Special handling for Google models which can't handle parallel tool calls
        if (
            model_type == "OpenAI"
            and is_google_model(model)
            and message.get("role") == "assistant"
            and isinstance(message.get("tool_calls"), list)
            and len(message["tool_calls"]) > 1
            and tools
            and len(tools) > 0
        ):
            first_tool_call = message["tool_calls"][0]
            first_tool_response = execute_tool_for_id(
                first_tool_call,
                tools,
                message_context,
            )
            # Add the first assistant message with only the first tool call
            transformed_messages.append({
                **message_base,
                "tool_calls": [first_tool_call],
            })
            # Add the first tool response if there is one
            if first_tool_response:
                transformed_messages.append(first_tool_response)
            # Process remaining tool calls in sequence, interlacing assistant and tool messages
            for tool_call in message["tool_calls"][1:]:
                # Add an empty assistant message with the next tool call
                transformed_messages.append({
                    "role": "assistant",
                    "content": "",
                    "tool_calls": [tool_call],
                })
                # Process this tool call and add the response
                tool_response = execute_tool_for_id(tool_call, tools, message_context)
                if tool_response:
                    transformed_messages.append(tool_response)
            # Skip the normal message processing since we've already handled everything
            continue

        tool_responses = []
        if model_type == "OpenAI" and message.get("tool_calls"):
            for tool_call in message["tool_calls"]:
                tool_responses.append(
                    execute_tool_for_id(
                        tool_call,
                        tools or [],
                        message_context,
                        messages,
                    )
                )

        # Inline tool results if required
        if (
            tool_responses
            and model_type == "OpenAI"
            and requires_inline_tool_results(model)
        ):
            tool_responses_string = create_inline_tool_results_content(tool_responses)
            if isinstance(message_base["content"], list):
                message_base["content"].append({
                    "type": "text",
                    "text": tool_responses_string,
                })
            else:
                message_base["content"] = f'{message_base["content"]} {tool_responses_string}'
            tool_responses = []

        # Mistral ordering if required
        if (
            tool_responses
            and model_type == "OpenAI"
            and requires_mistral_ordering(model)
        ):
            extra_mistral_message_arr.append({
                "role": "assistant",
                "content": "",
                "tool_calls": openai_extras.get("tool_calls"),
            })
            for tool_response in tool_responses:
                extra_mistral_message_arr.append(tool_response)
            tool_responses = []
            openai_extras["tool_calls"] = None

        # Ensure only tool calls have a tool_call_id
        if (
            model_type == "OpenAI"
            and message.get("role") != "tool"
            and openai_extras.get("tool_call_id")
        ):
            openai_extras["tool_call_id"] = None

        # Compose the final message(s)
        transformed_messages.extend(
            extra_mistral_message_arr +
            [
                {
                    **message_base,
                    **openai_extras,
                }
            ] +
            [tr for tr in tool_responses if tr]
        )

    return transformed_messages
