import pandas as pd
from networkx.algorithms.similarity import optimize_graph_edit_distance
import numpy as np
from graph_analysis import plan_to_dag, get_num_nodes, get_num_edges, validate_dag
from bert_score import score
import ast
import signal

def node_similarity(planA, planB):
    '''
    Compute node similarity between two plans.
    '''
    nodes_a = " ".join([node[1].get('name', "") for node in planA.nodes(data=True)])
    nodes_b = " ".join([node[1].get('name', "") for node in planB.nodes(data=True)])
    P, R, F1 = score([nodes_a], [nodes_b], lang="en")
    return F1.item()

def timeout_handler(signum, frame):
    raise TimeoutError("Function took too long!")

def compute_ged(G1, G2, timeout=120):
    '''
    Compute Graph Edit Distance.
    Utilize optimized graph edit distance -- nearest approximation of GED for larger graphs.
    '''
    signal.signal(signal.SIGALRM, timeout_handler)
    signal.alarm(timeout)  # Set timeout

    try:
        ged_iter = optimize_graph_edit_distance(G1, G2)
        result = list(ged_iter)
        if result:
            return result[-1]  
        else:
            return float('inf')  
    except TimeoutError:
        if ged_iter:
            return next(ged_iter, float('inf'))
        else:
            return float('inf')
    finally:
        signal.alarm(0)  

def senstitivity_analysis(conversation, planA, planB):
    '''
    Sensitivity computes:
        - Δ # of nodes between two plans
        - Δ # of edges between two plans
        - Semantic Distance between two plans (= 1 - BertScore)
        - Graph Edit Distance (GED) between two plans 
    '''
    diff_num_nodes = get_num_nodes(planA) - get_num_nodes(planB)
    diff_num_edges = get_num_edges(planA) - get_num_edges(planB)
    edit_distance = compute_ged(planA, planB, timeout=60)

    bert_score = node_similarity(planA, planB)

    return {
        "diff_num_nodes": diff_num_nodes,
        "diff_num_edges": diff_num_edges,
        "edit_distance": edit_distance,
        "semantic_distance": 1 - bert_score,
    }

def main(input_file, output_file):
    df = pd.read_json(input_file)
    senstitivity = []
    for i, row in df.iterrows():
        planA = plan_to_dag(ast.literal_eval(row['plan_from_conversation']))
        planB = plan_to_dag(ast.literal_eval(row['plan_from_rewrite']))

        if not validate_dag(planA):
            raise Exception("Invalid DAG")

        if not validate_dag(planB):
            raise Exception("Invalid DAG")

        eval = senstitivity_analysis(row['conversation'], planA, planB)
        senstitivity.append(eval)
    
    df['senstitivity'] = senstitivity
    df.to_json(output_file, orient='records', indent=2)
 
input_file = # Enter input file ""
output_file = # Enter output file ""

main(input_file, output_file)