#%%
import requests
from tqdm import tqdm
import os

url = "http://images.cocodataset.org/zips/train2014.zip"

def download_file(url, output_folder="."):
    local_filename = os.path.join(output_folder, url.split('/')[-1])
    with requests.get(url, stream=True) as r:
        r.raise_for_status()
        total_size = int(r.headers.get('content-length', 0))
        with open(local_filename, 'wb') as f, tqdm(
            total=total_size, unit='B', unit_scale=True, desc=local_filename
        ) as pbar:
            for chunk in r.iter_content(chunk_size=8192):
                if chunk:
                    f.write(chunk)
                    pbar.update(len(chunk))
    return local_filename

if __name__ == "__main__":
    os.makedirs("downloads", exist_ok=True)
    download_file(url, output_folder="downloads")

#%%
import json
import os
import cv2
import numpy as np
from matplotlib import pyplot as plt
import random

json_path = 'downloads/cocotext.v2.json'  
image_folder = 'downloads/train2014'  # Path to the folder containing images

with open(json_path, 'r') as f:
    data = json.load(f)

anns = data['anns']

candidates = [
    ann for ann in anns.values()
    if ann['legibility'] == 'legible'
    and ann['utf8_string'].strip()
    and ann.get('area', 0) >= 1000
]

if not candidates:
    raise ValueError("No annotation found with area >= 1000 and legible utf8_string.")

target_ann = random.choice(candidates)
target_image_id = target_ann['image_id']
target_image_name = f"COCO_train2014_{target_image_id:012d}.jpg"
print(f"Selected image: {target_image_name} (annotation id: {target_ann['id']}, text: {target_ann['utf8_string']})")

image_path = os.path.join(image_folder, target_image_name)
image = cv2.imread(image_path)
if image is None:
    raise FileNotFoundError(f"Image not found: {image_path}")

matching_anns = [ann for ann in anns.values() if ann['image_id'] == target_image_id]
matching_anns_sorted = sorted(matching_anns, key=lambda ann: ann.get('area', 0), reverse=True)
top_anns = matching_anns_sorted[:3]

for ann in top_anns:
    x, y, w, h = map(int, ann['bbox'])
    cv2.rectangle(image, (x, y), (x + w, y + h), color=(0, 255, 0), thickness=2)
    if ann['legibility'] == 'legible' and ann['utf8_string'].strip():
        cv2.putText(image, ann['utf8_string'], (x, y - 5),
                    fontFace=cv2.FONT_HERSHEY_SIMPLEX, fontScale=0.5,
                    color=(0, 0, 255), thickness=1)

image_rgb = cv2.cvtColor(image, cv2.COLOR_BGR2RGB)
plt.figure(figsize=(12, 8))
plt.imshow(image_rgb)
plt.title(f"Top 3 Annotations for {target_image_name}")
plt.axis('off')
plt.show()

#%%
import json
import matplotlib.pyplot as plt
from matplotlib.patches import Polygon
from PIL import Image
import numpy as np
import os
import random

annotation_path = "downloads/cocotext.v2.json"  
image_folder = "downloads/train2014"         

with open(annotation_path, "r") as f:
    data = json.load(f)

filtered_anns = [
    ann for ann in data["anns"].values()
    if ann["legibility"] == "legible" and ann["language"] == "english"
]
selected_ann = random.choice(filtered_anns)
image_id = selected_ann["image_id"]
utf8_string = selected_ann["utf8_string"]
mask_coords = np.array(selected_ann["mask"]).reshape(-1, 2)
image_filename = f"COCO_train2014_{image_id:012d}.jpg"
image_path = os.path.join(image_folder, image_filename)

if not os.path.exists(image_path):
    raise FileNotFoundError(f"Image not found: {image_path}")

image = Image.open(image_path)

plt.figure(figsize=(10, 10))
plt.imshow(image)
ax = plt.gca()

polygon = Polygon(mask_coords, closed=True, edgecolor='red', facecolor='none', linewidth=2)
ax.add_patch(polygon)

x, y = mask_coords[0]
ax.text(x, y - 5, utf8_string, color='red', fontsize=14)

plt.axis('off')
plt.title(f"Image ID: {image_id} - Word: '{utf8_string}'")
plt.show()

#%%
import os
import json
import cv2
import numpy as np
from PIL import Image

json_path = "downloads/cocotext.v2.json"
img_dir = "downloads/train2014"
resize_size = (384, 384)
area_thresholds = [500, 800, 1000, 1200, 1500, 2000]
samples_per_threshold = 30

with open(json_path, 'r') as f:
    data = json.load(f)

filtered_anns = [
    ann for ann in data["anns"].values()
    if ann["legibility"] == "legible" and ann["language"] == "english" and ann.get("area", 0) > 0
]

all_boxes = []

print(f"{len(filtered_anns)}")

for ann in filtered_anns:
    image_id = ann["image_id"]
    image_file = f"COCO_train2014_{image_id:012d}.jpg"
    image_path = os.path.join(img_dir, image_file)
    if not os.path.exists(image_path):
        continue

    try:
        with Image.open(image_path) as img:
            w_orig, h_orig = img.size
            scale_w = resize_size[0] / w_orig
            scale_h = resize_size[1] / h_orig
    except:
        continue

    scaled_area = ann["area"] * scale_w * scale_h
    bbox = ann["bbox"]
    scaled_bbox = [bbox[0]*scale_w, bbox[1]*scale_h, bbox[2]*scale_w, bbox[3]*scale_h]

    all_boxes.append({
        "image_id": image_id,
        "image_path": image_path,
        "word": ann.get("utf8_string", "").strip(),
        "area": scaled_area,
        "bbox": scaled_bbox
    })

for thr in area_thresholds:
    selected = sorted([x for x in all_boxes if x["area"] >= thr], key=lambda x: x["area"])[:samples_per_threshold]
    save_dir = f"cocotext_filtered_area_{thr}"
    os.makedirs(save_dir, exist_ok=True)

    metadata_records = []

    for i, item in enumerate(selected):
        img = cv2.imread(item["image_path"])
        if img is None:
            continue
        img_resized = cv2.resize(img, resize_size)
        x, y, w, h = map(int, item["bbox"])
        word = item["word"]
        area = item["area"]

        padded_img = np.zeros((414, 384, 3), dtype=np.uint8)
        padded_img[:384, :, :] = img_resized
        cv2.rectangle(padded_img, (x, y), (x + w, y + h), (0, 0, 255), 2)

        if word:
            cv2.putText(padded_img, word, (x, max(15, y - 5)), cv2.FONT_HERSHEY_SIMPLEX, 0.5, (255, 255, 255), 1)
        cv2.putText(padded_img, f"Area: {area:.1f}", (10, 404), cv2.FONT_HERSHEY_SIMPLEX, 0.6, (255, 255, 255), 2)

        out_path = os.path.join(save_dir, f"{i:02d}_{item['image_id']}.jpg")
        cv2.imwrite(out_path, padded_img)

        metadata_records.append({
            "image_id": item["image_id"],
            "word": word,
            "area": round(area, 2),
            "bbox": [round(x_, 2) for x_ in item["bbox"]],
            "image_file": os.path.basename(out_path)
        })

    jsonl_path = os.path.join(save_dir, "metadata.jsonl")
    with open(jsonl_path, "w") as f:
        for record in metadata_records:
            f.write(json.dumps(record) + "\n")
