#%%
import json
import os
import cv2
import random
import numpy as np
import math

json_files = [
    "textvqa_data/TextVQA_Rosetta_OCR_v0.2_train.json",
    "textvqa_data/TextVQA_Rosetta_OCR_v0.2_val.json",
    "textvqa_data/TextVQA_Rosetta_OCR_v0.2_test.json"
]

image_dirs = {
    "train": "textvqa_data/train_images",
    "test": "textvqa_data/test_images"
}

save_dir = "one_bbx_score_yaw_resized"
os.makedirs(save_dir, exist_ok=True)

def get_rotated_box_from_bbox(bbox, scale):
    x = bbox["top_left_x"] * scale
    y = bbox["top_left_y"] * scale
    w = bbox["width"] * scale
    h = bbox["height"] * scale
    yaw = math.radians(float(bbox.get("yaw", 0.0)))
    dx_w = w * math.cos(yaw)
    dy_w = w * math.sin(yaw)
    dx_h = -h * math.sin(yaw)
    dy_h = h * math.cos(yaw)
    pts = [
        (x, y),
        (x + dx_w, y + dy_w),
        (x + dx_w + dx_h, y + dy_w + dy_h),
        (x + dx_h, y + dy_h)
    ]
    return np.array([
        [int(px), int(py)]
        for (px, py) in pts
    ], dtype=np.int32)

combined_data = []
image_id_to_folder = {}

for json_path in json_files:
    folder = "test" if "test" in json_path.lower() else "train"
    with open(json_path, "r") as f:
        data = json.load(f)
        for item in data.get("data", []):
            combined_data.append(item)
            image_id_to_folder[item["image_id"]] = folder

eligible = [entry for entry in combined_data if "ocr_info" in entry and len(entry["ocr_info"]) > 0]
print("Total eligible entries:", len(eligible))

random.seed(42)
sampled = random.sample(eligible, min(40, len(eligible)))

for item in sampled:
    image_id = item["image_id"]
    ocr_infos = item["ocr_info"]

    scored_boxes = []
    for ann in ocr_infos:
        bbox = ann["bounding_box"]
        word = ann["word"]
        w = bbox["width"] * 384
        h = bbox["height"] * 384
        area = w * h
        # score = area / max(len(word), 1)
        score = area
        scored_boxes.append((bbox, word, area, score))

    top1 = sorted(scored_boxes, key=lambda x: x[3], reverse=True)[:1]

    filename = f"{image_id}.jpg"
    folder = image_id_to_folder.get(image_id, "train")
    image_path = os.path.join(image_dirs[folder], filename)

    if not os.path.exists(image_path):
        print(f"Image not found: {image_path}")
        continue

    img = cv2.imread(image_path)
    if img is None:
        print(f"Failed to read image: {image_path}")
        continue

    img_resized = cv2.resize(img, (384, 384))
    padded_img = np.zeros((414, 384, 3), dtype=np.uint8)
    padded_img[:384, :, :] = img_resized

    for i, (bbox, word, area, score) in enumerate(top1):
        box = get_rotated_box_from_bbox(bbox, scale=384)
        cv2.polylines(padded_img, [box], isClosed=True, color=(0, 0, 255), thickness=1)

        x_disp = int(bbox["top_left_x"] * 384)
        y_disp = int(bbox["top_left_y"] * 384)
        yaw = math.radians(float(bbox.get("yaw", 0.0)))
        p1 = (x_disp, y_disp)
        p2 = (int(x_disp + 40 * math.cos(yaw)), int(y_disp + 40 * math.sin(yaw)))
        cv2.line(padded_img, p1, p2, (255, 0, 0), 2)

        cv2.putText(padded_img, word, (x_disp, max(y_disp - 5, 15)), cv2.FONT_HERSHEY_SIMPLEX, 0.5, (0, 0, 255), 1)
        cv2.putText(padded_img, f"S{i+1}:{score:.1f}", (10 + i * 120, 404), cv2.FONT_HERSHEY_SIMPLEX, 0.6, (255, 255, 255), 2)

    out_path = os.path.join(save_dir, f"{image_id}.jpg")
    cv2.imwrite(out_path, padded_img)
    print(f"{image_id} | score = {score:.1f}")

#%%
#%%
import json
import os
import cv2
import random
import numpy as np
import math

json_files = [
    "textvqa_data/TextVQA_Rosetta_OCR_v0.2_train.json",
    "textvqa_data/TextVQA_Rosetta_OCR_v0.2_val.json",
    "textvqa_data/TextVQA_Rosetta_OCR_v0.2_test.json"
]

image_dirs = {
    "train": "textvqa_data/train_images",
    "test": "textvqa_data/test_images"
}

save_dir = "filtered_bbx_1000_min10"
os.makedirs(save_dir, exist_ok=True)

def get_rotated_box_from_bbox(bbox, scale):
    x = bbox["top_left_x"] * scale
    y = bbox["top_left_y"] * scale
    w = bbox["width"] * scale
    h = bbox["height"] * scale
    yaw = math.radians(float(bbox.get("yaw", 0.0)))
    dx_w = w * math.cos(yaw)
    dy_w = w * math.sin(yaw)
    dx_h = -h * math.sin(yaw)
    dy_h = h * math.cos(yaw)
    pts = [
        (x, y),
        (x + dx_w, y + dy_w),
        (x + dx_w + dx_h, y + dy_w + dy_h),
        (x + dx_h, y + dy_h)
    ]
    return np.array([
        [int(px), int(py)]
        for (px, py) in pts
    ], dtype=np.int32)

combined_data = []
image_id_to_folder = {}

for json_path in json_files:
    folder = "test" if "test" in json_path.lower() else "train"
    with open(json_path, "r") as f:
        data = json.load(f)
        for item in data.get("data", []):
            combined_data.append(item)
            image_id_to_folder[item["image_id"]] = folder

eligible = [entry for entry in combined_data if "ocr_info" in entry and len(entry["ocr_info"]) > 0]
print("Total eligible entries:", len(eligible))

all_filtered_boxes = []

for item in eligible:
    image_id = item["image_id"]
    folder = image_id_to_folder.get(image_id, "train")
    image_path = os.path.join(image_dirs[folder], f"{image_id}.jpg")
    if not os.path.exists(image_path):
        continue
    img = cv2.imread(image_path)
    if img is None:
        continue
    img_resized = cv2.resize(img, (384, 384))

    for ann in item["ocr_info"]:
        bbox = ann["bounding_box"]
        word = ann["word"]
        w = bbox["width"] * 384
        h = bbox["height"] * 384
        area = w * h
        if area <= 100:
            continue
        all_filtered_boxes.append({
            "image_id": image_id,
            "folder": folder,
            "image": img_resized,
            "bbox": bbox,
            "word": word,
            "area": area
        })

sorted_boxes = sorted(all_filtered_boxes, key=lambda x: x["area"])[:10]

for idx, item in enumerate(sorted_boxes):
    image_id = item["image_id"]
    bbox = item["bbox"]
    word = item["word"]
    area = item["area"]
    img_copy = item["image"].copy()

    padded_img = np.zeros((414, 384, 3), dtype=np.uint8)
    padded_img[:384, :, :] = img_copy

    box = get_rotated_box_from_bbox(bbox, scale=384)
    cv2.polylines(padded_img, [box], isClosed=True, color=(0, 0, 255), thickness=1)

    x_disp = int(bbox["top_left_x"] * 384)
    y_disp = int(bbox["top_left_y"] * 384)
    yaw = math.radians(float(bbox.get("yaw", 0.0)))
    p1 = (x_disp, y_disp)
    p2 = (int(x_disp + 40 * math.cos(yaw)), int(y_disp + 40 * math.sin(yaw)))
    cv2.line(padded_img, p1, p2, (255, 0, 0), 2)

    cv2.putText(padded_img, word, (x_disp, max(y_disp - 5, 15)), cv2.FONT_HERSHEY_SIMPLEX, 0.5, (0, 0, 255), 1)
    cv2.putText(padded_img, f"Area: {area:.1f}", (10, 404), cv2.FONT_HERSHEY_SIMPLEX, 0.6, (255, 255, 255), 2)

    out_path = os.path.join(save_dir, f"{idx:02d}_{image_id}.jpg")
    cv2.imwrite(out_path, padded_img)
    print(f"{idx+1}. {image_id} | area = {area:.1f}")
