import os
import json
import glob

# === Configuration ===
ROOT = "/fs/scratch/PAS2099/Jiacheng/Cure_or/01_no_challenge"
SUBDIRS = ["white/iPhone", "texture1/iPhone", "texture2/iPhone"]
MAPPING_TXT = "/fs/scratch/PAS2099/Jiacheng/Cure_or/category_mapping.txt"
OUTPUT_JSON = "/fs/scratch/PAS2099/Jiacheng/Cure_or/output/cure_or_general_complex.json"

# === Orientation ID to label mapping ===
ORIENT_MAP = {
    "1": "front",
    "2": "left",
    "3": "back",
    "4": "right",
    "5": "top"
}

# === Question template including 9 options (A–I) ===
QUESTION_TEMPLATE = (
    "From the perspective of the camera, which orientation is the {category} in the photo facing? "
    "A.front B.front right C.right D.back right E.back F.back left G.left H.front left I.top\n"
    "Answer with the option's letter and word from the given choices directly."
)

# === Step 1: Load the objectID → category_name mapping from TXT ===
id2name = {}
with open(MAPPING_TXT, "r") as f:
    for line in f:
        line = line.strip()
        if not line:
            continue
        parts = line.split(" ", 1)
        if len(parts) != 2:
            continue
        obj_id, name = parts
        id2name[obj_id] = name

# === Step 2: Traverse each subdirectory and build JSON entries ===
entries = []
for sub in SUBDIRS:
    folder = os.path.join(ROOT, sub)
    pattern = os.path.join(folder, "*.[jJpP][pPnN][gG]")
    for filepath in glob.glob(pattern):
        fname = os.path.basename(filepath)
        name, _ = os.path.splitext(fname)
        parts = name.split("_")
        # Expecting format: backgroundID_deviceID_objectOrientationID_objectID_challengeType_challengeLevel
        if len(parts) < 6:
            continue
        _, _, orient_id, obj_id, _, _ = parts
        orient = ORIENT_MAP.get(orient_id)
        if orient is None:
            continue

        # Lookup category name by objectID
        category = id2name.get(obj_id, "unknown")

        # Build the benchmark-style entry
        entry = {
            "type": "general_complex",
            "domain": "real",
            "base_dataset": "Cure_Or",
            "image": fname,
            "original_label": orient,
            "question": QUESTION_TEMPLATE.format(category=category),
            "label": orient,
            "category_name": category
        }
        entries.append(entry)

# === Step 3: Save all entries into the output JSON file ===
os.makedirs(os.path.dirname(OUTPUT_JSON), exist_ok=True)
with open(OUTPUT_JSON, "w") as f:
    json.dump(entries, f, indent=2, ensure_ascii=False)

print(f"✅ Done! Generated {len(entries)} entries saved to:\n   {OUTPUT_JSON}")
