#%%
import pandas as pd
import matplotlib.pyplot as plt
import matplotlib.patches as patches
from PIL import Image
import requests
from io import BytesIO
import ast
import random
import os

# === Config ===
csv_path = "lvis_pairwise_spatial_relations.csv"
num_to_plot = 5
random.seed(42)

# === Load CSV ===
df = pd.read_csv(csv_path)
sampled_rows = df.sample(n=min(num_to_plot, len(df)), random_state=42)

for _, row in sampled_rows.iterrows():
    image_url = row["image_url"]

    # Load image: try URL first, then local path
    try:
        if image_url.startswith("http"):
            response = requests.get(image_url, timeout=10)
            img = Image.open(BytesIO(response.content)).convert("RGB")
        else:
            img = Image.open(image_url).convert("RGB")
    except Exception as e:
        print(f"❌ Failed to load image: {image_url} — {e}")
        continue

    # Parse bbox and center values
    arche_bbox = ast.literal_eval(row["arche_bbox"])
    target_bbox = ast.literal_eval(row["target_bbox"])
    arche_center = ast.literal_eval(row["arche_center"])
    target_center = ast.literal_eval(row["target_center"])

    # Plot
    fig, ax = plt.subplots(1, figsize=(8, 6))
    ax.imshow(img)

    # Draw arche bbox (red)
    x, y, w, h = arche_bbox
    rect = patches.Rectangle((x, y), w, h,
                             linewidth=2, edgecolor="red", facecolor='none')
    ax.add_patch(rect)
    ax.text(x, y - 5, f"Arche: {row['arche_object']}",
            color="red", fontsize=12, weight="bold")

    # Draw target bbox (blue)
    x, y, w, h = target_bbox
    rect = patches.Rectangle((x, y), w, h,
                             linewidth=2, edgecolor="blue", facecolor='none')
    ax.add_patch(rect)
    ax.text(x, y - 5, f"Target: {row['target_object']}",
            color="blue", fontsize=12, weight="bold")

    # Show title with spatial relation
    ax.set_title(f"📍 Spatial: {row['spatial_answer']} | 🖼 {image_url}")
    ax.axis("off")
    plt.tight_layout()
    plt.show()

# %%
