import random, os
import torch
import numpy as np
import statistics
import math

class EarlyStopMonitor(object):
  def __init__(self, max_round=3, higher_better=True, tolerance=1e-3):
    self.max_round = max_round
    self.num_round = 0

    self.epoch_count = 0
    self.best_epoch = 0

    self.last_best = None
    self.higher_better = higher_better
    self.tolerance = tolerance

  def early_stop_check(self, curr_val):
    if not self.higher_better:
      curr_val *= -1
    if self.last_best is None:
      self.last_best = curr_val
    elif (curr_val - self.last_best) / np.abs(self.last_best) > self.tolerance:
      self.last_best = curr_val
      self.num_round = 0
      self.best_epoch = self.epoch_count
    else:
      self.num_round += 1
    self.epoch_count += 1
    return self.num_round >= self.max_round

class RandEdgeSampler(object):
  def __init__(self, src_list, dst_list):
    src_list = np.flip(np.concatenate(src_list))
    dst_list = np.flip(np.concatenate(dst_list))
    self.src_list, src_idx = np.unique(src_list, return_index=True)
    self.dst_list, dst_idx = np.unique(dst_list, return_index=True)

  def sample(self, size):
    src_index = np.random.randint(0, len(self.src_list), size)
    dst_index = np.random.randint(0, len(self.dst_list), size)
    return self.src_list[src_index], self.dst_list[dst_index]
  

def set_random_seed(seed):
  torch.manual_seed(seed)
  torch.cuda.manual_seed_all(seed)
  torch.backends.cudnn.deterministic = True
  torch.backends.cudnn.benchmark = False
  np.random.seed(seed)
  random.seed(seed)
  os.environ['PYTHONHASHSEED'] = str(seed)


def nat_results(logger, arr, name):
  logger.info(name + " " + str(arr))
  logger.info("Mean " + str(100 * statistics.mean(arr)))
  logger.info("Standard deviation " + str(statistics.pstdev(arr)))
  logger.info("95% " + str(1.96 * 100 * statistics.pstdev(arr) / math.sqrt(len(arr))))
  logger.info("--------")


def save_model(model, n_hop, epoch):
  for i in range(n_hop + 1):
    torch.save(model.neighborhood_store[i], model.get_ngh_store_path(epoch, i))
  torch.save(model.state_dict(), model.get_checkpoint_path(epoch))
  torch.save(model.self_rep, model.get_self_rep_path(epoch))
  torch.save(model.prev_raw, model.get_prev_raw_path(epoch))

def load_model(model, n_hop, epoch, seed):
  checkpoint_path = model.get_checkpoint_path(epoch)
  model.load_state_dict(torch.load(checkpoint_path))
  best_ngh_store = []
  model.clear_store()
  for i in range(n_hop + 1):
    best_ngh_store_path = model.get_ngh_store_path(epoch, i)
    best_ngh_store.append(torch.load(best_ngh_store_path))
  model.set_neighborhood_store(best_ngh_store)
  best_self_rep_path = model.get_self_rep_path(epoch)
  best_prev_raw_path = model.get_prev_raw_path(epoch)
  best_self_rep = torch.load(best_self_rep_path)
  best_prev_raw = torch.load(best_prev_raw_path)
  model.set_self_rep(best_self_rep, best_prev_raw)
  model.set_seed(seed)