from functools import partial

import torch
from torch import nn
from torch.nn import functional as F
from torch_geometric.nn import global_add_pool, global_mean_pool
import e3nn

from models.mace_modules.blocks import RadialEmbeddingBlock
import models.layers.gvp_layer as gvp


class BaseMLP(nn.Module):
    def __init__(self, input_dim, hidden_dim, output_dim, activation=nn.SiLU(), residual=False, last_act=False):
        super(BaseMLP, self).__init__()
        self.residual = residual
        if residual:
            assert output_dim == input_dim
        self.mlp = nn.Sequential(
            nn.Linear(input_dim, hidden_dim),
            activation,
            nn.Linear(hidden_dim, output_dim),
            activation if last_act else nn.Identity()
        )
    def forward(self, x):
        return x + self.mlp(x) if self.residual else self.mlp(x)
    
class NodeColor(nn.Module):
    def __init__(self, hidden_dim, color_type='center_radius', max_ell=6, activation=nn.SiLU()):
        super().__init__()
        MLP = partial(BaseMLP, hidden_dim=hidden_dim, activation=activation)
        if color_type == 'center_radius':
            self.mlp_node_feat = MLP(input_dim=1, output_dim=hidden_dim)
        elif color_type == 'tp':
            sh_irreps = e3nn.o3.Irreps.spherical_harmonics(max_ell)
            self.spherical_harmonics = e3nn.o3.SphericalHarmonics(
                sh_irreps, normalize=True, normalization="norm"
            )
            self.tp = e3nn.o3.FullyConnectedTensorProduct(sh_irreps, sh_irreps, f'{max_ell + 1}x0e',  shared_weights=False)
            
            self.mlp_sh_coff = MLP(input_dim=hidden_dim, output_dim=self.tp.weight_numel)
            self.mlp_node_feat = MLP(input_dim=max_ell + 1, output_dim=hidden_dim)
        self.color_type = color_type
        
    def forward(self, node_feat, node_pos, batch):
        center = global_mean_pool(node_pos, batch)
        pos = node_pos - center[batch]
        
        if self.color_type == 'center_radius':
            scalar = torch.norm(pos, dim=1, keepdim=True)
        elif self.color_type == 'tp':
            sh = self.spherical_harmonics(pos)
            global_sh = global_mean_pool(sh, batch)
            scalar = self.tp(sh, global_sh[batch], self.mlp_sh_coff(node_feat))
            
        return self.mlp_node_feat(scalar)

class VirtualNode(nn.Module):
    def __init__(self, num_vn=4, length=5.0, hidden_dim=64, activation=nn.SiLU()):
        super().__init__()
        self.num_vn = num_vn
        self.length = length
        self.get_vn_pos = e3nn.o3.FullyConnectedTensorProduct(
            '1x1o', '1x0e', f'{num_vn}x1o', shared_weights=False
        )
        
        MLP = partial(BaseMLP, hidden_dim=hidden_dim, activation=activation)
        self.mlp_vec_coff = MLP(input_dim=hidden_dim, output_dim=self.get_vn_pos.weight_numel)
    
    def forward(self, node_feat, node_pos, batch):
        center = global_mean_pool(node_pos, batch)
        pos = node_pos - center[batch]
        one = torch.ones([pos.size(0), 1], device=pos.device)

        vn_pos = global_mean_pool(
            self.get_vn_pos(pos, one, self.mlp_vec_coff(node_feat)), batch
        )
        vn_pos = vn_pos.view(-1, self.num_vn, 3)
        vn_pos = vn_pos / (torch.norm(vn_pos, dim=2, keepdim=True) + 1e-3) * self.length
        vn_pos = vn_pos.view(-1, self.num_vn * 3)
        vn_pos = vn_pos + center.repeat(1, self.num_vn)
        
        return vn_pos
    
class NodeFeatByVN(nn.Module):
    def __init__(self, num_vn=4, hidden_dim=64, activation=nn.SiLU()):
        super().__init__()
        self.num_vn = num_vn
        
        MLP = partial(BaseMLP, hidden_dim=hidden_dim, activation=activation)
        self.mlp_node_feat = MLP(input_dim=num_vn ** 2, output_dim=hidden_dim)
        
    def forward(self, node_feat, node_pos, vn_pos, batch):
        info_vec = node_pos.repeat(1, self.num_vn) - vn_pos[batch]
        info_vec = info_vec.view(node_pos.size(0), self.num_vn, 3)
        info_scalar = torch.cdist(info_vec, info_vec).view(node_pos.size(0), self.num_vn ** 2)
        # info_scalar = info_scalar / (torch.norm(info_scalar, dim=1, keepdim=True) + 1e-3)
        
        return self.mlp_node_feat(info_scalar)
    
class GVPGNNModel_cpl(torch.nn.Module):
    """
    GVP-GNN model from "Equivariant Graph Neural Networks for 3D Macromolecular Structure".
    """
    def __init__(
        self,
        r_max: float = 10.0,
        num_bessel: int = 8,
        num_polynomial_cutoff: int = 5,
        num_layers: int = 5,
        in_dim=1,
        out_dim=1,
        num_vn: int = 4,
        s_dim: int = 128,
        v_dim: int = 16,
        s_dim_edge: int = 32,
        v_dim_edge: int = 1,
        pool: str = "sum",
        residual: bool = True,
        equivariant_pred: bool = False
    ):
        """
        Initializes an instance of the GVPGNNModel class with the provided parameters.

        Parameters:
        - r_max (float): Maximum distance for Bessel basis functions (default: 10.0)
        - num_bessel (int): Number of Bessel basis functions (default: 8)
        - num_polynomial_cutoff (int): Number of polynomial cutoff basis functions (default: 5)
        - num_layers (int): Number of layers in the model (default: 5)
        - in_dim (int): Input dimension of the model (default: 1)
        - out_dim (int): Output dimension of the model (default: 1)
        - s_dim (int): Dimension of the node state embeddings (default: 128)
        - v_dim (int): Dimension of the node vector embeddings (default: 16)
        - s_dim_edge (int): Dimension of the edge state embeddings (default: 32)
        - v_dim_edge (int): Dimension of the edge vector embeddings (default: 1)
        - pool (str): Global pooling method to be used (default: "sum")
        - residual (bool): Whether to use residual connections (default: True)
        - equivariant_pred (bool): Whether it is an equivariant prediction task (default: False)
        """
        super().__init__()
        
        self.r_max = r_max
        self.num_layers = num_layers
        self.equivariant_pred = equivariant_pred
        self.s_dim = s_dim
        self.v_dim = v_dim
        
        activations = (F.relu, None)
        _DEFAULT_V_DIM = (s_dim, v_dim)
        _DEFAULT_E_DIM = (s_dim_edge, v_dim_edge) 

        # Node embedding
        self.emb_in = torch.nn.Embedding(in_dim, s_dim)
        self.num_vn = num_vn
        self.node_color = NodeColor(hidden_dim=s_dim)
        self.vn = VirtualNode(num_vn=num_vn, hidden_dim=s_dim)
        self.node_feat_by_vn = NodeFeatByVN(num_vn=num_vn, hidden_dim=s_dim)
        
        self.W_v = torch.nn.Sequential(
            gvp.LayerNorm((s_dim, 0)),
            gvp.GVP((s_dim, 0), _DEFAULT_V_DIM,
                activations=(None, None), vector_gate=True)
        )

        # Edge embedding
        self.radial_embedding = RadialEmbeddingBlock(
            r_max=r_max,
            num_bessel=num_bessel,
            num_polynomial_cutoff=num_polynomial_cutoff,
        )
        self.W_e = torch.nn.Sequential(
            gvp.LayerNorm((self.radial_embedding.out_dim, 1)),
            gvp.GVP((self.radial_embedding.out_dim, 1), _DEFAULT_E_DIM, 
                activations=(None, None), vector_gate=True)
        )
        
        # Stack of GNN layers
        self.layers = torch.nn.ModuleList(
            gvp.GVPConvLayer(
                _DEFAULT_V_DIM, _DEFAULT_E_DIM, 
                activations=activations, vector_gate=True,
                residual=residual
            ) 
            for _ in range(num_layers)
        )
        
        # Global pooling/readout function
        self.pool = {"mean": global_mean_pool, "sum": global_add_pool}[pool]

        if self.equivariant_pred:
            # Linear predictor for equivariant tasks using geometric features
            self.pred = torch.nn.Linear(s_dim + v_dim * 3, out_dim)
        else:
            # MLP predictor for invariant tasks using only scalar features
            self.pred = torch.nn.Sequential(
                torch.nn.Linear(s_dim, s_dim),
                torch.nn.ReLU(),
                torch.nn.Linear(s_dim, out_dim)
            )
    
    def forward(self, batch):

        # Edge features
        vectors = batch.pos[batch.edge_index[0]] - batch.pos[batch.edge_index[1]]  # [n_edges, 3]
        lengths = torch.linalg.norm(vectors, dim=-1, keepdim=True)  # [n_edges, 1]
        
        h_V = self.emb_in(batch.atoms)  # (n,) -> (n, d)
        pos = batch.pos  # (n, 3)
        
        h_V = h_V + self.node_color(h_V, pos, batch.batch)
        vn_pos = self.vn(h_V, pos, batch.batch)
        h_V = h_V + self.node_feat_by_vn(h_V, pos, vn_pos, batch.batch)
        
        
        h_E = (
            self.radial_embedding(lengths), 
            torch.nan_to_num(torch.div(vectors, lengths)).unsqueeze_(-2)
        )

        h_V = self.W_v(h_V)
        h_E = self.W_e(h_E)
    
        for layer in self.layers:
            h_V = layer(h_V, batch.edge_index, h_E)

        out = self.pool(gvp._merge(*h_V), batch.batch)  # (n, d) -> (batch_size, d)
        
        if not self.equivariant_pred:
            # Select only scalars for invariant prediction
            out = out[:,:self.s_dim]
        
        return self.pred(out)  # (batch_size, out_dim)