from typing import Optional, Tuple, Union, Dict, List
from functools import partial

import torch
from torch import nn, LongTensor, Tensor
from torch.nn import functional as F
from torch_geometric.data import Data
from torch_geometric.nn import global_add_pool, global_mean_pool
from torch_scatter import scatter
import e3nn

from models.mace_modules.blocks import RadialEmbeddingBlock
from models.mace_modules.irreps_tools import irreps2gate

class BaseMLP(nn.Module):
    def __init__(
        self, 
        input_dim: int, 
        hidden_dim: int, 
        output_dim: int, 
        activation: nn.Module = nn.SiLU(),
        norm: Optional[nn.Module] = None,
        residual: bool = False, 
        last_act: bool = False,
    ) -> None:
        super(BaseMLP, self).__init__()
        self.residual = residual
        if residual:
            assert output_dim == input_dim
        self.mlp = nn.Sequential(
            nn.Linear(input_dim, hidden_dim),
            nn.Identity() if norm is None else norm(hidden_dim),
            activation,
            nn.Linear(hidden_dim, output_dim),
            nn.Identity() if norm is None else norm(output_dim),
            activation if last_act else nn.Identity()
        )

    def forward(self, x):
        return x + self.mlp(x) if self.residual else self.mlp(x)
    
class VNInitial(nn.Module):
    def __init__(
        self,
        hidden_dim: int = 64,
        vn_channel: int = 4,
        edge_attr_dim: int = 2,
        activation: nn.Module = nn.SiLU(),
        norm: Optional[nn.Module] = None,
        device: str = 'cpu',
    ) -> None:
        super().__init__()
        self.vn_channel = vn_channel

        self.diff_vec_coff = e3nn.o3.FullyConnectedTensorProduct(
            '2x1o', '1x0e', f'{self.vn_channel}x1o', shared_weights=False
        )
        
        MLP = partial(BaseMLP, hidden_dim=hidden_dim, activation=activation, norm=norm)
        self.mlp_feat_init = MLP(input_dim=hidden_dim, output_dim=hidden_dim, last_act=True)
        self.mlp_msg = MLP(input_dim=2 * hidden_dim + edge_attr_dim + 4, output_dim=hidden_dim)
        self.mlp_diff_vel_coff = MLP(input_dim=hidden_dim, output_dim=self.diff_vec_coff.weight_numel)
        self.mlp_vn_feat = MLP(input_dim=hidden_dim, output_dim=hidden_dim)
        
        
    def forward(self, node_feat, node_pos, node_vel, edge_index, edge_attr):
        vn_feat = self.mlp_feat_init(node_feat)
        vn_pos = node_pos.repeat(1, self.vn_channel)
        
        row, col = edge_index
        diff_pos = node_pos[row] - node_pos[col]
        diff_vel = node_vel[row] - node_vel[col]
        
        diff_vec = torch.cat([diff_pos, diff_vel], dim=1)
        ip = torch.einsum('bij,bkj->bik', diff_vec.view(-1, 2, 3), diff_vec.view(-1, 2, 3)).view(-1, 4)
        ip = ip / (torch.norm(ip) + 1e-3)
        diff_vec = diff_vec / (torch.norm(ip) + 1e-3)
        
        one = torch.ones([diff_vec.size(0), 1], device=diff_vec.device)
        msg = torch.cat([node_feat[row], node_feat[col], edge_attr, ip], dim=1)
        msg = self.mlp_msg(msg)
        diff_vec = self.diff_vec_coff(diff_vec, one, self.mlp_diff_vel_coff(msg))

        agg_feat = scatter(src=msg, index=row, dim=0, dim_size=node_feat.size(0), reduce='mean')
        agg_vec = scatter(src=diff_vec, index=row, dim=0, dim_size=node_feat.size(0), reduce='mean')
        vn_feat = vn_feat + self.mlp_vn_feat(agg_feat)
        vn_pos = vn_pos + agg_vec
        return vn_feat, vn_pos
    
class VNLayer(nn.Module):
    def __init__(
        self,
        hidden_dim: int = 64,
        hidden_irreps: e3nn.o3.Irreps = '8x0e+8x1o+8x2e',
        scalar_dim: int = 8,
        vn_channel: int = 4,
        edge_attr_dim: int = 2,
        activation: nn.Module = nn.SiLU(),
        norm: Optional[nn.Module] = None,
        device: str = 'cpu',
    ) -> None:
        super().__init__()
        self.vn_channel = vn_channel
        self.scalar_dim = scalar_dim

        self.diff_pos_vr_coff = e3nn.o3.FullyConnectedTensorProduct(
            f'{self.vn_channel}x1o', '1x0e', '1x1o', shared_weights=False
        )
        self.diff_pos_rv_coff = e3nn.o3.FullyConnectedTensorProduct(
            f'{self.vn_channel}x1o', '1x0e', f'{self.vn_channel}x1o', shared_weights=False
        )
        self.hidden_irreps = hidden_irreps
        self.hidden_dim = hidden_dim
        self.get_scalar = e3nn.o3.Linear(hidden_irreps, f'{hidden_dim}x0e')
        
        MLP = partial(BaseMLP, hidden_dim=hidden_dim, activation=activation, norm=norm)
        self.mlp_com_msg = MLP(input_dim=2 * hidden_dim + vn_channel ** 2, output_dim=hidden_dim)
        self.mlp_diff_pos_vr_coff = MLP(input_dim=hidden_dim, output_dim=self.diff_pos_vr_coff.weight_numel)
        self.mlp_diff_pos_rv_coff = MLP(input_dim=hidden_dim, output_dim=self.diff_pos_rv_coff.weight_numel)
        self.mlp_feat = MLP(input_dim=hidden_dim, output_dim=hidden_dim)
        self.mlp_feat_change_dim = MLP(input_dim=hidden_dim, output_dim=scalar_dim)
        self.mlp_vn_feat = MLP(input_dim=hidden_dim, output_dim=hidden_dim)
        
    def forward(self, node_feat, node_pos, vn_feat, vn_pos, edge_index, edge_attr):
        node_feat_scalar = self.get_scalar(node_feat)
        row, col = edge_index
        diff_pos_vr = node_pos.repeat(1, self.vn_channel)[row] - vn_pos[col]
        ip = torch.einsum('bij,bkj->bik', diff_pos_vr.view(-1, self.vn_channel, 3), diff_pos_vr.view(-1, self.vn_channel, 3)).view(-1, self.vn_channel ** 2)
        ip = ip / (torch.norm(ip) + 1e-3)
        diff_pos_vr = diff_pos_vr / (torch.norm(ip) + 1e-3)
        diff_pos_rv = - diff_pos_vr
        
        com_msg_vr = torch.cat([node_feat_scalar[row], vn_feat[col], ip], dim=1)
        com_msg_vr = self.mlp_com_msg(com_msg_vr)
        
        one_vr = torch.ones([diff_pos_vr.size(0), 1], device=diff_pos_vr.device)
        diff_pos_vr = self.diff_pos_vr_coff(diff_pos_vr, one_vr, self.mlp_diff_pos_vr_coff(com_msg_vr)) 
        
        one_rv = torch.ones([diff_pos_rv.size(0), 1], device=diff_pos_rv.device)
        diff_pos_rv = self.diff_pos_rv_coff(diff_pos_rv, one_rv, self.mlp_diff_pos_rv_coff(com_msg_vr)) 
        
        dim_size = node_feat.size(0)
        agg_feat_vr = scatter(src=com_msg_vr, index=row, dim=0, dim_size=dim_size, reduce='mean')
        agg_pos_vr = scatter(src=diff_pos_vr, index=row, dim=0, dim_size=dim_size, reduce='mean')
        agg_pos_rv = scatter(src=diff_pos_rv, index=row, dim=0, dim_size=dim_size, reduce='mean')
        
        node_feat_scalar = node_feat_scalar + self.mlp_feat(agg_feat_vr)
        node_feat_scalar = self.mlp_feat_change_dim(node_feat_scalar)
        node_feat = node_feat + F.pad(node_feat_scalar, (0, self.hidden_irreps.dim - self.scalar_dim))
        node_pos = node_pos + agg_pos_vr
        
        vn_feat = vn_feat + self.mlp_vn_feat(agg_feat_vr)
        vn_pos = vn_pos + agg_pos_rv
        return node_feat, node_pos, vn_feat, vn_pos
    

class TensorProductConvLayer(nn.Module):
    def __init__(
        self,
        in_irreps: e3nn.o3.Irreps,
        out_irreps: e3nn.o3.Irreps,
        sh_irreps: e3nn.o3.Irreps,
        hidden_dim: int = 64,
        edge_attr_dim: int = 2,
        activation: nn.Module = nn.SiLU(),
        norm: Optional[nn.Module] = None,
    ):
        super().__init__()

        self.tp = e3nn.o3.FullyConnectedTensorProduct(
            in_irreps, sh_irreps, out_irreps, shared_weights=False
        )
        self.re_scale = e3nn.o3.FullyConnectedTensorProduct(
            out_irreps, '1x0e', out_irreps, shared_weights=False
        )
        self.get_scalar = e3nn.o3.Linear(out_irreps, f'{hidden_dim}x0e')

        MLP = partial(BaseMLP, hidden_dim=hidden_dim, activation=activation, norm=norm)
        self.mlp_sh_coff = MLP(input_dim=edge_attr_dim, output_dim=self.tp.weight_numel)
        self.mlp_cpl_coff = MLP(input_dim=hidden_dim, output_dim=self.re_scale.weight_numel)
        
    def forward(self, node_feat, edge_index, edge_attr, edge_sh):
        col, row = edge_index
        msg = self.tp(node_feat[col], edge_sh, self.mlp_sh_coff(edge_attr))
        agg = scatter(src=msg, index=row, dim=0, dim_size=node_feat.size(0), reduce='mean')
        node_feat = node_feat + agg

        one = torch.ones([node_feat.size(0), 1], device=node_feat.device)
        node_feat = self.re_scale(node_feat, one, self.mlp_cpl_coff(self.get_scalar(agg)))

        return node_feat


class TFNModel_cpl_local(torch.nn.Module):
    """
    Tensor Field Network model from "Tensor Field Networks".
    """
    def __init__(
        self,
        r_max: float = 10.0,
        num_bessel: int = 8,
        num_polynomial_cutoff: int = 5,
        max_ell: int = 2,
        require_vel: bool = True,
        num_layer: int = 4,
        hidden_dim: int = 64,
        irreps_channels: int = 8,
        node_input_dim: int = 2,
        edge_attr_dim: int = 2,
        vn_channel: int = 4,
        activation: nn.Module = nn.SiLU(),
        norm: Optional[nn.Module] = None,
        device: str = 'cpu',
    ):
        super().__init__()
        self.vn_init = VNInitial(hidden_dim, vn_channel, edge_attr_dim)
        
        sh_irreps = e3nn.o3.Irreps.spherical_harmonics(max_ell)
        self.spherical_harmonics = e3nn.o3.SphericalHarmonics(
            sh_irreps, normalize=True, normalization="norm"
        )
        self.input_irreps = e3nn.o3.Irreps(f'{irreps_channels}x0e+1x1o') if require_vel else e3nn.o3.Irreps(f'{irreps_channels}x0e')
        self.hidden_irreps = (sh_irreps * irreps_channels).sort()[0].simplify()
        
        self.radial_embedding = RadialEmbeddingBlock(
            r_max=r_max,
            num_bessel=num_bessel,
            num_polynomial_cutoff=num_polynomial_cutoff,
        )
        
        self.embedding = nn.Linear(node_input_dim, irreps_channels)
        self.embedding_vn_init = nn.Linear(node_input_dim, hidden_dim)

        self.layers = torch.nn.ModuleList()
        for _ in range(num_layer):
            self.layers.append(
                TensorProductConvLayer(
                    in_irreps=self.hidden_irreps,
                    out_irreps=self.hidden_irreps,
                    sh_irreps=sh_irreps,
                    hidden_dim=hidden_dim,
                    edge_attr_dim=self.radial_embedding.out_dim + edge_attr_dim,
                    activation=activation,
                    norm=norm,
                )
            )

        self.vn_layers = torch.nn.ModuleList()
        for _ in range(num_layer):
            self.vn_layers.append(VNLayer(hidden_dim, self.hidden_irreps, irreps_channels, vn_channel, edge_attr_dim, activation, norm))
            
        self.get_delta_pos = e3nn.o3.Linear(self.hidden_irreps, '1x1o')

    
    def forward(self, data: Data):
        node_feat = self.embedding(data.node_feat)
        node_pos = data.node_pos
        node_vel = data.node_vel if 'node_vel' in data else (data.node_pos * 0).detach()
        node_feat = torch.cat([i for i in [node_feat, node_vel] if i is not None], dim=-1)

        edge_index = data.edge_index
        col, row = edge_index
        edge_attr = data.edge_attr if 'edge_attr' in data else None

        vn_feat, vn_pos = self.vn_init(self.embedding_vn_init(data.node_feat), node_pos, node_vel, edge_index, edge_attr)

        diff_pos = node_pos[col] - node_pos[row]
        dist = torch.norm(diff_pos, dim=1, keepdim=True)

        edge_attr = torch.cat([i for i in [
            edge_attr,
            self.radial_embedding(dist)
        ] if i is not None], dim=-1)
        edge_sh = self.spherical_harmonics(diff_pos)
        
        node_feat = F.pad(node_feat, (0, self.hidden_irreps.dim - self.input_irreps.dim))

        for layer, vn_layer in zip(self.layers, self.vn_layers):
            node_feat = layer(node_feat, edge_index, edge_attr, edge_sh)
            node_feat, node_pos, vn_feat, vn_pos = vn_layer(node_feat, node_pos, vn_feat, vn_pos, edge_index, edge_attr)
        node_pos = node_pos + self.get_delta_pos(node_feat)
        
        return node_pos
