function [L_hat, convergence_flag,  all_relRes, iterations__with_unchanged_lambda] = ...
    RGNMR(X,omega, rank, outliers_num, opts, test)

%% INPUT: 
% X - the observed matrix
% omega - list of pairs (i,j) of the observed entries
% rank - target rank of the underlying matrix
% outliers_num - upper bound on the number of outlires
% opts - options meta-variable (see set_deafult_options method for details)
% test - optinal, for testing 
%
%% OUTPUT:
% L_hat - estimate of the underlying matrix
% convergence_flag - indicating whether the algorithm converged
% all_relRes - list of the residual errors of the least-squares problem throughout the iterations
% number of itertion since Lambda changed


if nargin < 6
    test = false;
end

%% set RGNMR configuration
opts  = set_default_options(opts);

%% initalize RGNMR
[U, V, L_hat, Lambda] = init_RGNMR(opts.init_option, opts.init_U, opts.init_V, X, omega, rank, outliers_num, test);

%% define variables for later use 
[n1, n2] = size(X);   % number of rows and colums
colind_A = generate_sparse_matrix_indices(omega, rank, n2); % used when constructing the sparse matrix A

%% set variables before iterations
all_relRes = []; % stores the relRes of all iterations
convergence_flag = 0; % if true breaks from iteration 
LSQR_tol = opts.inner_init_tol; % the tolorence used by the LSQR solver
iterations__with_unchanged_lambda  = 0;
L_hat_diff = 1;

%% iterations
for iter = 1:opts.max_outer_iter
    L_hat_previous = L_hat;
    Lambda_previous = Lambda;
   
    %% solve the LSQR problem
    [U, V, L_hat, entriwise_residulas, all_relRes(iter), LSQR_iters_done] = solve_LSQR_problem(X, U, V,omega, Lambda, colind_A, LSQR_tol, opts.max_inner_iter);

    %%  update the estimate of the set of non corrupted entries represented by D
    Lambda = binary_weights(entriwise_residulas,  outliers_num, omega, test);
    
    %% check if Lambda changed 
    iterations__with_unchanged_lambda = (iterations__with_unchanged_lambda + 1) * isequal(Lambda, Lambda_previous);
    
    %% decrease the tolorence
    if opts.LSQR_smart_tol
        LSQR_tol = max(2*eps, all_relRes(iter)*1e-1);
    end
    
    %% report
    if opts.verbose
        L_hat_diff = norm(L_hat - L_hat_previous, 'fro') / norm(L_hat, 'fro');
        fprintf('[INSIDE RGNMR] iter %4d \t diff X_r %5d\t relRes %6d\n',...
            iter, L_hat_diff, all_relRes(iter));
    end
    %% check early convergence
    convergence_flag = check_early_convergence( iterations__with_unchanged_lambda, opts.stop_lambda_convergence, ...
                                                all_relRes, opts.stop_relRes, ...
                                                L_hat, L_hat_previous,  opts.stop_relDiff, ...
                                                opts.stop_relResDiff, iter, LSQR_iters_done, opts.verbose);

    %% if failed to solve lsqr problem start from random initialization
    if iter > 1 && LSQR_iters_done == 0
        [U, V, L_hat, D] = init_RGNMR(1, opts.init_U, opts.init_V, X, omega, rank, outliers_num, test);
        convergence_flag = false;
    end
    if convergence_flag
        break
    end   
end

%% return
[U_hat, lambda_hat, V_hat] = svds(L_hat ,rank);
L_hat = U_hat * lambda_hat * V_hat';
end
