
from omegaconf import OmegaConf
import wandb
import torch
import random
import numpy as np
import os

def load_config(config_file="configs/config.yaml"):
    """Load configuration file"""
    cfg = OmegaConf.load(config_file)
    return cfg


def set_seed(seed: int):
    """Set random seed for reproducibility."""
    random.seed(seed)
    np.random.seed(seed)
    torch.manual_seed(seed)
    torch.cuda.manual_seed_all(seed)

    # For deterministic behavior
    torch.backends.cudnn.deterministic = True
    torch.backends.cudnn.benchmark = False

    # Make hash seed fixed (affects Python hashing)
    os.environ["PYTHONHASHSEED"] = str(seed)

def setup_device(gpu):
    """Setup device configuration based on GPU availability."""
    device = torch.device("mps" if torch.has_mps else "cpu")
    device = torch.device(f"cuda:{gpu}" if torch.cuda.is_available() else "cpu")
    print(f"Using device: {device}")
    return device

def init_wandb(run_name, cfg):
    """Initialize the wandb tracking for a new run."""
    wandb.init(
        project="vlm_adapt",
        name=run_name,
        config=OmegaConf.to_container(cfg, resolve=True),
        reinit=True,  # Re-init in a loop
    )