# Tool-focused Cross-Pollination with Judgment Strategy

Having reviewed the files provided to you above, you are creating a new Claude Code agent primarily by combining successful elements from multiple existing agents using the three-artifact architecture.

## Context
You're evolving a database analysis system that consists of three distinct artifacts:
1. **eval_instructions.md** - Static SQL generation instructions passed directly to the eval model
2. **agent.md** - Database analysis agent that examines specific databases and runs tools
3. **tools/** - Python/shell scripts for database analysis

## Your Task
Create a new agent package by cross-pollinating successful patterns from the top-performing agents to achieve higher accuracy on databases you haven't seen yet.  

You may have worked on creating these agent packages before.  If so, feel free to use your knowledge and experience.

**Note:**  Although you are primarily using a cross-pollinating approach, you can use a new idea of your own if you think you see an opportunity.

## Required Output Structure

You must create the following files:

### 1. reasoning.md 
You are trying to achieve higher accuracy than any of the agents you are examining.  Please give your analysis of what to combine and why, based on, for instance:
- Identification of complementary strengths across different agents
- Analysis of which agent excels at which types of problems
- Specific combinations that could address observed weaknesses
- Hypotheses about synergistic combinations

### 2. eval_instructions.md 
Complete SQL generation instructions that combine the best patterns from multiple agents.   For example, you might want to:
- Merge effective column selection rules from different sources
- Combine complementary SQL pattern recognitions
- Integrate different error-avoidance strategies
- Synthesize output format requirements
- Unify successful approaches to evidence handling

Notes:  
- Write your own instructions, as appropriate, to achieve the goals laid out in reasoning.md
- On the other hand, feel free to copy (or copy and then modify) instructions from other agents if you see something that you like
- These instructions go DIRECTLY to the eval model


### 3. tools/ 
Analysis tools as Python (.py) or shell (.sh) scripts:
- You can use standard Python libraries + sqlite3
- You can also use any other Python library or any command-line tool if you see that it is currently installed
- You should analyze the specific database found at ./database.sqlite
- Generate database-specific guidance
- Output to stdout or files in ./tool_output/
- Include error handling
- Write your own scripts, as appropriate, to achieve the goals laid out in reasoning.md
- On the other hand, feel free to copy (or copy and then modify) scripts from other agents if you see something that you like
- The tools should incorporate the ideas you described in reasoning.md


When generating these tools, think about what information the eval model will need about this specific database to do its job.  Keep in mind what you told the eval model in eval_instructions.md.  You don't need to repeat that here.  Note that the eval model will not have any information about the database beyond the information that your analysis agent provides

In contrast to some prior approaches you might see, you will be seeking to implement the vision you laid out in reasoning.md primarily through the use of these tools.  You are permitted to leave some aspects of the DB Analysis up to the agent (discussed below), but your focus should be on providing the eval model with the necessary information by writing appropriate scripts

Example tool structure:
```python
#!/usr/bin/env python3
import sqlite3
import json

def analyze_database(db_path):
    # Your analysis logic
    results = {...}
    
    # Output to tool_output directory
    with open('tool_output/analysis.json', 'w') as f:
        json.dump(results, f, indent=2)
    
    print("Analysis complete - results in tool_output/analysis.json")

if __name__ == "__main__":
    analyze_database("database.sqlite")
```

### 4. agent.md
Database analysis agent with YAML frontmatter that MUST include these EXACT two fields:
```yaml
---
name: your-unique-agent-name-here
description: Brief description of the approach you are taking with the overall three-artifact package
---
```

Note 1: The agent name will be used to create a directory (with hyphens converted to underscores).
For evolved agents, the system will prefix your agent name with the iteration number (e.g., iter2_your_agent_name).

Note 2:  In line with previous instructions, feel free to write your own agent.md or to copy elements from the agent.md files of other agents as makes sense to achieve your goals

The agent and the tools together should document the database in a way that will help the eval model to generate accurate SQL.  When thinking about the agent, please consider the following:
- Your approach is to implement as much as possible through the use of the tools.  Don't instruct the agent to do something that you can accomplish with a script
- So you could decide that your instructions to the agent will be as simple as the below:

```
# Tool Runner Agent

Execute database analysis tools and return the results.

## Process

1. Run the main analysis tool: `python tools/analyze_database.py`
2. Read the complete output from `tool_output/analysis.json`
3. Return the entire contents as your output to `./output/agent_output.txt`

The tools will handle all database analysis. Your role is simply to execute them and pass through the results.
```

On the other hand, there might be some things that are better handled by the AI agent (which will generally be a Claude Code Sonnet-4 model):
- A parameterization of your script or some post-processing which is best handled by an AI
- Error handling or handling of unusual situations
- Tasks that are difficult to capture in a deterministic script and are best delegated to an AI agent

## Cross-Pollination Requirements

### 1. Identify Complementary Strengths
- Find agents that excel at different aspects:
  - One might handle JOINs well
  - Another might excel at aggregations
  - A third might be best at string matching
- Look for non-overlapping strengths to combine

### 2. Analyze Performance Patterns
- Review where each agent succeeds and fails
- Identify which components contribute to success:
  - Specific eval_instructions patterns
  - Particular analysis approaches in agent.md
  - Effective tool strategies
- Find synergies between different approaches

### 3. Strategic Combination
- **Don't just merge everything** - be selective
- Choose combinations that:
  - Address identified weaknesses
  - Don't conflict with each other
  - Create emergent capabilities
- Avoid a "kitchen sink" approach - each element should have a purpose

### 4. Design Synthesis
Based on your analysis:
- **eval_instructions.md**: Which SQL rules complement each other?
- **agent.md**: Which analysis strategies work together?
- **tools/**: Which automated analyses combine well?


## Success Metrics

The main success metric is that you want to achieve higher accuracy than any previous agent on databases you haven't seen yet.  You are seeking to accomplish that primarily through a cross-pollination strategy

Your cross-pollinated package should:
- Preserve good general-purpose SQL generation instructions (in eval_instructions.md)
- Provide rich database-specific context (in the output of your tools and agent.md)
- Combine strengths without inheriting weaknesses
- Address failure patterns from multiple agents
- Create emergent capabilities through combination
- Maintain clean separation of concerns
- Be more robust than any single predecessor

## Important Notes

- The final system prompt will be: [agent output] + [eval_instructions]
- Don't just concatenate - thoughtfully combine
- Test combinations for conflicts and redundancies
- Focus on complementary strengths, not just "more of everything"
- Start with 2-3 top performers and selectively combine



Remember: **Think harder** than you normally would about this. You are primarily using the cross-pollinating approach, but you can also use a new idea of your own if you think it is important for the overall goal of SQL generation.  Note that the cross-pollination technique you will be primarily using is about finding synergies between different successful approaches. The idea is not to include everything, but to create something better through strategic combination!