import os
from typing import Callable, Tuple, Optional, Dict, Any, List

from PIL import Image
import pandas as pd
import torch
from torchvision.datasets.utils import verify_str_arg
from torchvision.datasets.folder import default_loader

from cv_lib.utils import log_utils

from .classification_dataset import ClassificationDataset
from .utils import make_datafolder


MEAN = [0.485, 0.456, 0.406]
STD = [0.229, 0.224, 0.225]


class MiniImagenet(ClassificationDataset):
    """
    Image folder:
        ├── devkit
        │   └── data
        ├── train_set
        │   ├── n01440764
        │       ├── n01440764_10026.JPEG
        |   |   ├── ...
        │   ├── n01443537
        │   ├── ...
        ├── val_set
    """
    def __init__(
        self,
        root: str,
        split: str = "train",
        resize: Optional[Tuple[int]] = None,
        augmentations: Callable[[Image.Image, Dict[str, Any]], Tuple[Image.Image, Dict[str, Any]]] = None,
        make_partial: float = None,
        manual_classes_fp: str = None,
        use_real_class_name: bool = False
    ):
        """
        Args:
            root: root to MiniImagenet folder
            split: split of dataset, i.e., `train` and `val`
            resize: all images will be resized to given size. If `None`, all images will not be resized
            manual_classes_fp: csv file with all manual classes in a column with name "classes"
            use_real_class_name: if True, use real class name for as label info, e.g.,
                "kit_fox" rather than "n02119789"
        """
        super().__init__(resize, augmentations)
        self.root = os.path.expanduser(root)
        verify_str_arg(split, "split", ("train", "val"))
        self.split = split
        self.data_folder = os.path.join(self.root, self.split)
        self.meta_folder = os.path.join(self.root, "devkit", "data")
        self.use_real_class_name = use_real_class_name

        self.logger = log_utils.get_master_logger("Imagenet")

        classes = None
        if manual_classes_fp:
            df = pd.read_csv(manual_classes_fp)
            classes = list(df["classes"])
        self._init_dataset(make_partial, classes)

    def _init_dataset(self, make_partial: float, manual_classes: List[str]):
        self.dataset_mean = MEAN
        self.dataset_std = STD
        self.logger.info("Reading dataset folder...")
        self.instances, self.label_info, self.label_map = make_datafolder(
            self.data_folder,
            make_partial,
            manual_classes
        )
        if self.use_real_class_name:
            self.label_info = {k: REAL_LABEL_MAP[v] for k, v in self.label_info.items()}
            self.label_map = {REAL_LABEL_MAP[k]: v for k, v in self.label_map.items()}

    def __len__(self):
        return len(self.instances)

    def get_image(self, index: int) -> Image:
        image_fp = os.path.join(self.data_folder, self.instances[index][0])
        image = default_loader(image_fp)
        return image

    def get_annotation(self, index: int) -> Dict[str, Any]:
        label = self.instances[index][1]
        annot = dict(label=torch.tensor(label))
        return annot


# collected from https://gist.github.com/aaronpolhamus/964a4411c0906315deb9f4a3723aac57
REAL_LABEL_MAP = {
    "n02119789": "kit_fox",
    "n02100735": "English_setter",
    "n02110185": "Siberian_husky",
    "n02096294": "Australian_terrier",
    "n02102040": "English_springer",
    "n02066245": "grey_whale",
    "n02509815": "lesser_panda",
    "n02124075": "Egyptian_cat",
    "n02417914": "ibex",
    "n02123394": "Persian_cat",
    "n02125311": "cougar",
    "n02423022": "gazelle",
    "n02346627": "porcupine",
    "n02077923": "sea_lion",
    "n02110063": "malamute",
    "n02447366": "badger",
    "n02109047": "Great_Dane",
    "n02089867": "Walker_hound",
    "n02102177": "Welsh_springer_spaniel",
    "n02091134": "whippet",
    "n02092002": "Scottish_deerhound",
    "n02071294": "killer_whale",
    "n02442845": "mink",
    "n02504458": "African_elephant",
    "n02092339": "Weimaraner",
    "n02098105": "soft-coated_wheaten_terrier",
    "n02096437": "Dandie_Dinmont",
    "n02114712": "red_wolf",
    "n02105641": "Old_English_sheepdog",
    "n02128925": "jaguar",
    "n02091635": "otterhound",
    "n02088466": "bloodhound",
    "n02096051": "Airedale",
    "n02117135": "hyena",
    "n02138441": "meerkat",
    "n02097130": "giant_schnauzer",
    "n02493509": "titi",
    "n02457408": "three-toed_sloth",
    "n02389026": "sorrel",
    "n02443484": "black-footed_ferret",
    "n02110341": "dalmatian",
    "n02089078": "black-and-tan_coonhound",
    "n02086910": "papillon",
    "n02445715": "skunk",
    "n02093256": "Staffordshire_bullterrier",
    "n02113978": "Mexican_hairless",
    "n02106382": "Bouvier_des_Flandres",
    "n02441942": "weasel",
    "n02113712": "miniature_poodle",
    "n02113186": "Cardigan",
    "n02105162": "malinois",
    "n02415577": "bighorn",
    "n02356798": "fox_squirrel",
    "n02488702": "colobus",
    "n02123159": "tiger_cat",
    "n02098413": "Lhasa",
    "n02422699": "impala",
    "n02114855": "coyote",
    "n02094433": "Yorkshire_terrier",
    "n02111277": "Newfoundland",
    "n02132136": "brown_bear",
    "n02119022": "red_fox",
    "n02091467": "Norwegian_elkhound",
    "n02106550": "Rottweiler",
    "n02422106": "hartebeest",
    "n02091831": "Saluki",
    "n02120505": "grey_fox",
    "n02104365": "schipperke",
    "n02086079": "Pekinese",
    "n02112706": "Brabancon_griffon",
    "n02098286": "West_Highland_white_terrier",
    "n02095889": "Sealyham_terrier",
    "n02484975": "guenon",
    "n02137549": "mongoose",
    "n02500267": "indri",
    "n02129604": "tiger",
    "n02090721": "Irish_wolfhound",
    "n02396427": "wild_boar",
    "n02108000": "EntleBucher",
    "n02391049": "zebra",
    "n02412080": "ram",
    "n02108915": "French_bulldog",
    "n02480495": "orangutan",
    "n02110806": "basenji",
    "n02128385": "leopard",
    "n02107683": "Bernese_mountain_dog",
    "n02085936": "Maltese_dog",
    "n02094114": "Norfolk_terrier",
    "n02087046": "toy_terrier",
    "n02100583": "vizsla",
    "n02096177": "cairn",
    "n02494079": "squirrel_monkey",
    "n02105056": "groenendael",
    "n02101556": "clumber",
    "n02123597": "Siamese_cat",
    "n02481823": "chimpanzee",
    "n02105505": "komondor",
    "n02088094": "Afghan_hound",
    "n02085782": "Japanese_spaniel",
    "n02489166": "proboscis_monkey",
    "n02364673": "guinea_pig",
    "n02114548": "white_wolf",
    "n02134084": "ice_bear",
    "n02480855": "gorilla",
    "n02090622": "borzoi",
    "n02113624": "toy_poodle",
    "n02093859": "Kerry_blue_terrier",
    "n02403003": "ox",
    "n02097298": "Scotch_terrier",
    "n02108551": "Tibetan_mastiff",
    "n02493793": "spider_monkey",
    "n02107142": "Doberman",
    "n02096585": "Boston_bull",
    "n02107574": "Greater_Swiss_Mountain_dog",
    "n02107908": "Appenzeller",
    "n02086240": "Shih-Tzu",
    "n02102973": "Irish_water_spaniel",
    "n02112018": "Pomeranian",
    "n02093647": "Bedlington_terrier",
    "n02397096": "warthog",
    "n02437312": "Arabian_camel",
    "n02483708": "siamang",
    "n02097047": "miniature_schnauzer",
    "n02106030": "collie",
    "n02099601": "golden_retriever",
    "n02093991": "Irish_terrier",
    "n02110627": "affenpinscher",
    "n02106166": "Border_collie",
    "n02326432": "hare",
    "n02108089": "boxer",
    "n02097658": "silky_terrier",
    "n02088364": "beagle",
    "n02111129": "Leonberg",
    "n02100236": "German_short-haired_pointer",
    "n02486261": "patas",
    "n02115913": "dhole",
    "n02486410": "baboon",
    "n02487347": "macaque",
    "n02099849": "Chesapeake_Bay_retriever",
    "n02108422": "bull_mastiff",
    "n02104029": "kuvasz",
    "n02492035": "capuchin",
    "n02110958": "pug",
    "n02099429": "curly-coated_retriever",
    "n02094258": "Norwich_terrier",
    "n02099267": "flat-coated_retriever",
    "n02395406": "hog",
    "n02112350": "keeshond",
    "n02109961": "Eskimo_dog",
    "n02101388": "Brittany_spaniel",
    "n02113799": "standard_poodle",
    "n02095570": "Lakeland_terrier",
    "n02128757": "snow_leopard",
    "n02101006": "Gordon_setter",
    "n02115641": "dingo",
    "n02097209": "standard_schnauzer",
    "n02342885": "hamster",
    "n02097474": "Tibetan_terrier",
    "n02120079": "Arctic_fox",
    "n02095314": "wire-haired_fox_terrier",
    "n02088238": "basset",
    "n02408429": "water_buffalo",
    "n02133161": "American_black_bear",
    "n02328150": "Angora",
    "n02410509": "bison",
    "n02492660": "howler_monkey",
    "n02398521": "hippopotamus",
    "n02112137": "chow",
    "n02510455": "giant_panda",
    "n02093428": "American_Staffordshire_terrier",
    "n02105855": "Shetland_sheepdog",
    "n02111500": "Great_Pyrenees",
    "n02085620": "Chihuahua",
    "n02123045": "tabby",
    "n02490219": "marmoset",
    "n02099712": "Labrador_retriever",
    "n02109525": "Saint_Bernard",
    "n02454379": "armadillo",
    "n02111889": "Samoyed",
    "n02088632": "bluetick",
    "n02090379": "redbone",
    "n02443114": "polecat",
    "n02361337": "marmot",
    "n02105412": "kelpie",
    "n02483362": "gibbon",
    "n02437616": "llama",
    "n02107312": "miniature_pinscher",
    "n02325366": "wood_rabbit",
    "n02091032": "Italian_greyhound",
    "n02129165": "lion",
    "n02102318": "cocker_spaniel",
    "n02100877": "Irish_setter",
    "n02074367": "dugong",
    "n02504013": "Indian_elephant",
    "n02363005": "beaver",
    "n02102480": "Sussex_spaniel",
    "n02113023": "Pembroke",
    "n02086646": "Blenheim_spaniel",
    "n02497673": "Madagascar_cat",
    "n02087394": "Rhodesian_ridgeback",
    "n02127052": "lynx",
    "n02116738": "African_hunting_dog",
    "n02488291": "langur",
    "n02091244": "Ibizan_hound",
    "n02114367": "timber_wolf",
    "n02130308": "cheetah",
    "n02089973": "English_foxhound",
    "n02105251": "briard",
    "n02134418": "sloth_bear",
    "n02093754": "Border_terrier",
    "n02106662": "German_shepherd",
    "n02444819": "otter",
    "n01882714": "koala",
    "n01871265": "tusker",
    "n01872401": "echidna",
    "n01877812": "wallaby",
    "n01873310": "platypus",
    "n01883070": "wombat",
    "n04086273": "revolver",
    "n04507155": "umbrella",
    "n04147183": "schooner",
    "n04254680": "soccer_ball",
    "n02672831": "accordion",
    "n02219486": "ant",
    "n02317335": "starfish",
    "n01968897": "chambered_nautilus",
    "n03452741": "grand_piano",
    "n03642806": "laptop",
    "n07745940": "strawberry",
    "n02690373": "airliner",
    "n04552348": "warplane",
    "n02692877": "airship",
    "n02782093": "balloon",
    "n04266014": "space_shuttle",
    "n03344393": "fireboat",
    "n03447447": "gondola",
    "n04273569": "speedboat",
    "n03662601": "lifeboat",
    "n02951358": "canoe",
    "n04612504": "yawl",
    "n02981792": "catamaran",
    "n04483307": "trimaran",
    "n03095699": "container_ship",
    "n03673027": "liner",
    "n03947888": "pirate",
    "n02687172": "aircraft_carrier",
    "n04347754": "submarine",
    "n04606251": "wreck",
    "n03478589": "half_track",
    "n04389033": "tank",
    "n03773504": "missile",
    "n02860847": "bobsled",
    "n03218198": "dogsled",
    "n02835271": "bicycle-built-for-two",
    "n03792782": "mountain_bike",
    "n03393912": "freight_car",
    "n03895866": "passenger_car",
    "n02797295": "barrow",
    "n04204347": "shopping_cart",
    "n03791053": "motor_scooter",
    "n03384352": "forklift",
    "n03272562": "electric_locomotive",
    "n04310018": "steam_locomotive",
    "n02704792": "amphibian",
    "n02701002": "ambulance",
    "n02814533": "beach_wagon",
    "n02930766": "cab",
    "n03100240": "convertible",
    "n03594945": "jeep",
    "n03670208": "limousine",
    "n03770679": "minivan",
    "n03777568": "Model_T",
    "n04037443": "racer",
    "n04285008": "sports_car",
    "n03444034": "go-kart",
    "n03445924": "golfcart",
    "n03785016": "moped",
    "n04252225": "snowplow",
    "n03345487": "fire_engine",
    "n03417042": "garbage_truck",
    "n03930630": "pickup",
    "n04461696": "tow_truck",
    "n04467665": "trailer_truck",
    "n03796401": "moving_van",
    "n03977966": "police_van",
    "n04065272": "recreational_vehicle",
    "n04335435": "streetcar",
    "n04252077": "snowmobile",
    "n04465501": "tractor",
    "n03776460": "mobile_home",
    "n04482393": "tricycle",
    "n04509417": "unicycle",
    "n03538406": "horse_cart",
    "n03599486": "jinrikisha",
    "n03868242": "oxcart",
    "n02804414": "bassinet",
    "n03125729": "cradle",
    "n03131574": "crib",
    "n03388549": "four-poster",
    "n02870880": "bookcase",
    "n03018349": "china_cabinet",
    "n03742115": "medicine_chest",
    "n03016953": "chiffonier",
    "n04380533": "table_lamp",
    "n03337140": "file",
    "n03891251": "park_bench",
    "n02791124": "barber_chair",
    "n04429376": "throne",
    "n03376595": "folding_chair",
    "n04099969": "rocking_chair",
    "n04344873": "studio_couch",
    "n04447861": "toilet_seat",
    "n03179701": "desk",
    "n03982430": "pool_table",
    "n03201208": "dining_table",
    "n03290653": "entertainment_center",
    "n04550184": "wardrobe",
    "n07742313": "Granny_Smith",
    "n07747607": "orange",
    "n07749582": "lemon",
    "n07753113": "fig",
    "n07753275": "pineapple",
    "n07753592": "banana",
    "n07754684": "jackfruit",
    "n07760859": "custard_apple",
    "n07768694": "pomegranate",
    "n12267677": "acorn",
    "n12620546": "hip",
    "n13133613": "ear",
    "n11879895": "rapeseed",
    "n12144580": "corn",
    "n12768682": "buckeye",
    "n03854065": "organ",
    "n04515003": "upright",
    "n03017168": "chime",
    "n03249569": "drum",
    "n03447721": "gong",
    "n03720891": "maraca",
    "n03721384": "marimba",
    "n04311174": "steel_drum",
    "n02787622": "banjo",
    "n02992211": "cello",
    "n04536866": "violin",
    "n03495258": "harp",
    "n02676566": "acoustic_guitar",
    "n03272010": "electric_guitar",
    "n03110669": "cornet",
    "n03394916": "French_horn",
    "n04487394": "trombone",
    "n03494278": "harmonica",
    "n03840681": "ocarina",
    "n03884397": "panpipe",
    "n02804610": "bassoon",
    "n03838899": "oboe",
    "n04141076": "sax",
    "n03372029": "flute",
    "n11939491": "daisy",
    "n12057211": "yellow_lady's_slipper",
    "n09246464": "cliff",
    "n09468604": "valley",
    "n09193705": "alp",
    "n09472597": "volcano",
    "n09399592": "promontory",
    "n09421951": "sandbar",
    "n09256479": "coral_reef",
    "n09332890": "lakeside",
    "n09428293": "seashore",
    "n09288635": "geyser",
    "n03498962": "hatchet",
    "n03041632": "cleaver",
    "n03658185": "letter_opener",
    "n03954731": "plane",
    "n03995372": "power_drill",
    "n03649909": "lawn_mower",
    "n03481172": "hammer",
    "n03109150": "corkscrew",
    "n02951585": "can_opener",
    "n03970156": "plunger",
    "n04154565": "screwdriver",
    "n04208210": "shovel",
    "n03967562": "plow",
    "n03000684": "chain_saw",
    "n01514668": "cock",
    "n01514859": "hen",
    "n01518878": "ostrich",
    "n01530575": "brambling",
    "n01531178": "goldfinch",
    "n01532829": "house_finch",
    "n01534433": "junco",
    "n01537544": "indigo_bunting",
    "n01558993": "robin",
    "n01560419": "bulbul",
    "n01580077": "jay",
    "n01582220": "magpie",
    "n01592084": "chickadee",
    "n01601694": "water_ouzel",
    "n01608432": "kite",
    "n01614925": "bald_eagle",
    "n01616318": "vulture",
    "n01622779": "great_grey_owl",
    "n01795545": "black_grouse",
    "n01796340": "ptarmigan",
    "n01797886": "ruffed_grouse",
    "n01798484": "prairie_chicken",
    "n01806143": "peacock",
    "n01806567": "quail",
    "n01807496": "partridge",
    "n01817953": "African_grey",
    "n01818515": "macaw",
    "n01819313": "sulphur-crested_cockatoo",
    "n01820546": "lorikeet",
    "n01824575": "coucal",
    "n01828970": "bee_eater",
    "n01829413": "hornbill",
    "n01833805": "hummingbird",
    "n01843065": "jacamar",
    "n01843383": "toucan",
    "n01847000": "drake",
    "n01855032": "red-breasted_merganser",
    "n01855672": "goose",
    "n01860187": "black_swan",
    "n02002556": "white_stork",
    "n02002724": "black_stork",
    "n02006656": "spoonbill",
    "n02007558": "flamingo",
    "n02009912": "American_egret",
    "n02009229": "little_blue_heron",
    "n02011460": "bittern",
    "n02012849": "crane",
    "n02013706": "limpkin",
    "n02018207": "American_coot",
    "n02018795": "bustard",
    "n02025239": "ruddy_turnstone",
    "n02027492": "red-backed_sandpiper",
    "n02028035": "redshank",
    "n02033041": "dowitcher",
    "n02037110": "oystercatcher",
    "n02017213": "European_gallinule",
    "n02051845": "pelican",
    "n02056570": "king_penguin",
    "n02058221": "albatross",
    "n01484850": "great_white_shark",
    "n01491361": "tiger_shark",
    "n01494475": "hammerhead",
    "n01496331": "electric_ray",
    "n01498041": "stingray",
    "n02514041": "barracouta",
    "n02536864": "coho",
    "n01440764": "tench",
    "n01443537": "goldfish",
    "n02526121": "eel",
    "n02606052": "rock_beauty",
    "n02607072": "anemone_fish",
    "n02643566": "lionfish",
    "n02655020": "puffer",
    "n02640242": "sturgeon",
    "n02641379": "gar",
    "n01664065": "loggerhead",
    "n01665541": "leatherback_turtle",
    "n01667114": "mud_turtle",
    "n01667778": "terrapin",
    "n01669191": "box_turtle",
    "n01675722": "banded_gecko",
    "n01677366": "common_iguana",
    "n01682714": "American_chameleon",
    "n01685808": "whiptail",
    "n01687978": "agama",
    "n01688243": "frilled_lizard",
    "n01689811": "alligator_lizard",
    "n01692333": "Gila_monster",
    "n01693334": "green_lizard",
    "n01694178": "African_chameleon",
    "n01695060": "Komodo_dragon",
    "n01704323": "triceratops",
    "n01697457": "African_crocodile",
    "n01698640": "American_alligator",
    "n01728572": "thunder_snake",
    "n01728920": "ringneck_snake",
    "n01729322": "hognose_snake",
    "n01729977": "green_snake",
    "n01734418": "king_snake",
    "n01735189": "garter_snake",
    "n01737021": "water_snake",
    "n01739381": "vine_snake",
    "n01740131": "night_snake",
    "n01742172": "boa_constrictor",
    "n01744401": "rock_python",
    "n01748264": "Indian_cobra",
    "n01749939": "green_mamba",
    "n01751748": "sea_snake",
    "n01753488": "horned_viper",
    "n01755581": "diamondback",
    "n01756291": "sidewinder",
    "n01629819": "European_fire_salamander",
    "n01630670": "common_newt",
    "n01631663": "eft",
    "n01632458": "spotted_salamander",
    "n01632777": "axolotl",
    "n01641577": "bullfrog",
    "n01644373": "tree_frog",
    "n01644900": "tailed_frog",
    "n04579432": "whistle",
    "n04592741": "wing",
    "n03876231": "paintbrush",
    "n03483316": "hand_blower",
    "n03868863": "oxygen_mask",
    "n04251144": "snorkel",
    "n03691459": "loudspeaker",
    "n03759954": "microphone",
    "n04152593": "screen",
    "n03793489": "mouse",
    "n03271574": "electric_fan",
    "n03843555": "oil_filter",
    "n04332243": "strainer",
    "n04265275": "space_heater",
    "n04330267": "stove",
    "n03467068": "guillotine",
    "n02794156": "barometer",
    "n04118776": "rule",
    "n03841143": "odometer",
    "n04141975": "scale",
    "n02708093": "analog_clock",
    "n03196217": "digital_clock",
    "n04548280": "wall_clock",
    "n03544143": "hourglass",
    "n04355338": "sundial",
    "n03891332": "parking_meter",
    "n04328186": "stopwatch",
    "n03197337": "digital_watch",
    "n04317175": "stethoscope",
    "n04376876": "syringe",
    "n03706229": "magnetic_compass",
    "n02841315": "binoculars",
    "n04009552": "projector",
    "n04356056": "sunglasses",
    "n03692522": "loupe",
    "n04044716": "radio_telescope",
    "n02879718": "bow",
    "n02950826": "cannon",
    "n02749479": "assault_rifle",
    "n04090263": "rifle",
    "n04008634": "projectile",
    "n03085013": "computer_keyboard",
    "n04505470": "typewriter_keyboard",
    "n03126707": "crane",
    "n03666591": "lighter",
    "n02666196": "abacus",
    "n02977058": "cash_machine",
    "n04238763": "slide_rule",
    "n03180011": "desktop_computer",
    "n03485407": "hand-held_computer",
    "n03832673": "notebook",
    "n06359193": "web_site",
    "n03496892": "harvester",
    "n04428191": "thresher",
    "n04004767": "printer",
    "n04243546": "slot",
    "n04525305": "vending_machine",
    "n04179913": "sewing_machine",
    "n03602883": "joystick",
    "n04372370": "switch",
    "n03532672": "hook",
    "n02974003": "car_wheel",
    "n03874293": "paddlewheel",
    "n03944341": "pinwheel",
    "n03992509": "potter's_wheel",
    "n03425413": "gas_pump",
    "n02966193": "carousel",
    "n04371774": "swing",
    "n04067472": "reel",
    "n04040759": "radiator",
    "n04019541": "puck",
    "n03492542": "hard_disc",
    "n04355933": "sunglass",
    "n03929660": "pick",
    "n02965783": "car_mirror",
    "n04258138": "solar_dish",
    "n04074963": "remote_control",
    "n03208938": "disk_brake",
    "n02910353": "buckle",
    "n03476684": "hair_slide",
    "n03627232": "knot",
    "n03075370": "combination_lock",
    "n03874599": "padlock",
    "n03804744": "nail",
    "n04127249": "safety_pin",
    "n04153751": "screw",
    "n03803284": "muzzle",
    "n04162706": "seat_belt",
    "n04228054": "ski",
    "n02948072": "candle",
    "n03590841": "jack-o'-lantern",
    "n04286575": "spotlight",
    "n04456115": "torch",
    "n03814639": "neck_brace",
    "n03933933": "pier",
    "n04485082": "tripod",
    "n03733131": "maypole",
    "n03794056": "mousetrap",
    "n04275548": "spider_web",
    "n01768244": "trilobite",
    "n01770081": "harvestman",
    "n01770393": "scorpion",
    "n01773157": "black_and_gold_garden_spider",
    "n01773549": "barn_spider",
    "n01773797": "garden_spider",
    "n01774384": "black_widow",
    "n01774750": "tarantula",
    "n01775062": "wolf_spider",
    "n01776313": "tick",
    "n01784675": "centipede",
    "n01990800": "isopod",
    "n01978287": "Dungeness_crab",
    "n01978455": "rock_crab",
    "n01980166": "fiddler_crab",
    "n01981276": "king_crab",
    "n01983481": "American_lobster",
    "n01984695": "spiny_lobster",
    "n01985128": "crayfish",
    "n01986214": "hermit_crab",
    "n02165105": "tiger_beetle",
    "n02165456": "ladybug",
    "n02167151": "ground_beetle",
    "n02168699": "long-horned_beetle",
    "n02169497": "leaf_beetle",
    "n02172182": "dung_beetle",
    "n02174001": "rhinoceros_beetle",
    "n02177972": "weevil",
    "n02190166": "fly",
    "n02206856": "bee",
    "n02226429": "grasshopper",
    "n02229544": "cricket",
    "n02231487": "walking_stick",
    "n02233338": "cockroach",
    "n02236044": "mantis",
    "n02256656": "cicada",
    "n02259212": "leafhopper",
    "n02264363": "lacewing",
    "n02268443": "dragonfly",
    "n02268853": "damselfly",
    "n02276258": "admiral",
    "n02277742": "ringlet",
    "n02279972": "monarch",
    "n02280649": "cabbage_butterfly",
    "n02281406": "sulphur_butterfly",
    "n02281787": "lycaenid",
    "n01910747": "jellyfish",
    "n01914609": "sea_anemone",
    "n01917289": "brain_coral",
    "n01924916": "flatworm",
    "n01930112": "nematode",
    "n01943899": "conch",
    "n01944390": "snail",
    "n01945685": "slug",
    "n01950731": "sea_slug",
    "n01955084": "chiton",
    "n02319095": "sea_urchin",
    "n02321529": "sea_cucumber",
    "n03584829": "iron",
    "n03297495": "espresso_maker",
    "n03761084": "microwave",
    "n03259280": "Dutch_oven",
    "n04111531": "rotisserie",
    "n04442312": "toaster",
    "n04542943": "waffle_iron",
    "n04517823": "vacuum",
    "n03207941": "dishwasher",
    "n04070727": "refrigerator",
    "n04554684": "washer",
    "n03133878": "Crock_Pot",
    "n03400231": "frying_pan",
    "n04596742": "wok",
    "n02939185": "caldron",
    "n03063689": "coffeepot",
    "n04398044": "teapot",
    "n04270147": "spatula",
    "n02699494": "altar",
    "n04486054": "triumphal_arch",
    "n03899768": "patio",
    "n04311004": "steel_arch_bridge",
    "n04366367": "suspension_bridge",
    "n04532670": "viaduct",
    "n02793495": "barn",
    "n03457902": "greenhouse",
    "n03877845": "palace",
    "n03781244": "monastery",
    "n03661043": "library",
    "n02727426": "apiary",
    "n02859443": "boathouse",
    "n03028079": "church",
    "n03788195": "mosque",
    "n04346328": "stupa",
    "n03956157": "planetarium",
    "n04081281": "restaurant",
    "n03032252": "cinema",
    "n03529860": "home_theater",
    "n03697007": "lumbermill",
    "n03065424": "coil",
    "n03837869": "obelisk",
    "n04458633": "totem_pole",
    "n02980441": "castle",
    "n04005630": "prison",
    "n03461385": "grocery_store",
    "n02776631": "bakery",
    "n02791270": "barbershop",
    "n02871525": "bookshop",
    "n02927161": "butcher_shop",
    "n03089624": "confectionery",
    "n04200800": "shoe_shop",
    "n04443257": "tobacco_shop",
    "n04462240": "toyshop",
    "n03388043": "fountain",
    "n03042490": "cliff_dwelling",
    "n04613696": "yurt",
    "n03216828": "dock",
    "n02892201": "brass",
    "n03743016": "megalith",
    "n02788148": "bannister",
    "n02894605": "breakwater",
    "n03160309": "dam",
    "n03000134": "chainlink_fence",
    "n03930313": "picket_fence",
    "n04604644": "worm_fence",
    "n04326547": "stone_wall",
    "n03459775": "grille",
    "n04239074": "sliding_door",
    "n04501370": "turnstile",
    "n03792972": "mountain_tent",
    "n04149813": "scoreboard",
    "n03530642": "honeycomb",
    "n03961711": "plate_rack",
    "n03903868": "pedestal",
    "n02814860": "beacon",
    "n07711569": "mashed_potato",
    "n07720875": "bell_pepper",
    "n07714571": "head_cabbage",
    "n07714990": "broccoli",
    "n07715103": "cauliflower",
    "n07716358": "zucchini",
    "n07716906": "spaghetti_squash",
    "n07717410": "acorn_squash",
    "n07717556": "butternut_squash",
    "n07718472": "cucumber",
    "n07718747": "artichoke",
    "n07730033": "cardoon",
    "n07734744": "mushroom",
    "n04209239": "shower_curtain",
    "n03594734": "jean",
    "n02971356": "carton",
    "n03485794": "handkerchief",
    "n04133789": "sandal",
    "n02747177": "ashcan",
    "n04125021": "safe",
    "n07579787": "plate",
    "n03814906": "necklace",
    "n03134739": "croquet_ball",
    "n03404251": "fur_coat",
    "n04423845": "thimble",
    "n03877472": "pajama",
    "n04120489": "running_shoe",
    "n03062245": "cocktail_shaker",
    "n03014705": "chest",
    "n03717622": "manhole_cover",
    "n03777754": "modem",
    "n04493381": "tub",
    "n04476259": "tray",
    "n02777292": "balance_beam",
    "n07693725": "bagel",
    "n03998194": "prayer_rug",
    "n03617480": "kimono",
    "n07590611": "hot_pot",
    "n04579145": "whiskey_jug",
    "n03623198": "knee_pad",
    "n07248320": "book_jacket",
    "n04277352": "spindle",
    "n04229816": "ski_mask",
    "n02823428": "beer_bottle",
    "n03127747": "crash_helmet",
    "n02877765": "bottlecap",
    "n04435653": "tile_roof",
    "n03724870": "mask",
    "n03710637": "maillot",
    "n03920288": "Petri_dish",
    "n03379051": "football_helmet",
    "n02807133": "bathing_cap",
    "n04399382": "teddy",
    "n03527444": "holster",
    "n03983396": "pop_bottle",
    "n03924679": "photocopier",
    "n04532106": "vestment",
    "n06785654": "crossword_puzzle",
    "n03445777": "golf_ball",
    "n07613480": "trifle",
    "n04350905": "suit",
    "n04562935": "water_tower",
    "n03325584": "feather_boa",
    "n03045698": "cloak",
    "n07892512": "red_wine",
    "n03250847": "drumstick",
    "n04192698": "shield",
    "n03026506": "Christmas_stocking",
    "n03534580": "hoopskirt",
    "n07565083": "menu",
    "n04296562": "stage",
    "n02869837": "bonnet",
    "n07871810": "meat_loaf",
    "n02799071": "baseball",
    "n03314780": "face_powder",
    "n04141327": "scabbard",
    "n04357314": "sunscreen",
    "n02823750": "beer_glass",
    "n13052670": "hen-of-the-woods",
    "n07583066": "guacamole",
    "n03637318": "lampshade",
    "n04599235": "wool",
    "n07802026": "hay",
    "n02883205": "bow_tie",
    "n03709823": "mailbag",
    "n04560804": "water_jug",
    "n02909870": "bucket",
    "n03207743": "dishrag",
    "n04263257": "soup_bowl",
    "n07932039": "eggnog",
    "n03786901": "mortar",
    "n04479046": "trench_coat",
    "n03873416": "paddle",
    "n02999410": "chain",
    "n04367480": "swab",
    "n03775546": "mixing_bowl",
    "n07875152": "potpie",
    "n04591713": "wine_bottle",
    "n04201297": "shoji",
    "n02916936": "bulletproof_vest",
    "n03240683": "drilling_platform",
    "n02840245": "binder",
    "n02963159": "cardigan",
    "n04370456": "sweatshirt",
    "n03991062": "pot",
    "n02843684": "birdhouse",
    "n03482405": "hamper",
    "n03942813": "ping-pong_ball",
    "n03908618": "pencil_box",
    "n03902125": "pay-phone",
    "n07584110": "consomme",
    "n02730930": "apron",
    "n04023962": "punching_bag",
    "n02769748": "backpack",
    "n10148035": "groom",
    "n02817516": "bearskin",
    "n03908714": "pencil_sharpener",
    "n02906734": "broom",
    "n03788365": "mosquito_net",
    "n02667093": "abaya",
    "n03787032": "mortarboard",
    "n03980874": "poncho",
    "n03141823": "crutch",
    "n03976467": "Polaroid_camera",
    "n04264628": "space_bar",
    "n07930864": "cup",
    "n04039381": "racket",
    "n06874185": "traffic_light",
    "n04033901": "quill",
    "n04041544": "radio",
    "n07860988": "dough",
    "n03146219": "cuirass",
    "n03763968": "military_uniform",
    "n03676483": "lipstick",
    "n04209133": "shower_cap",
    "n03782006": "monitor",
    "n03857828": "oscilloscope",
    "n03775071": "mitten",
    "n02892767": "brassiere",
    "n07684084": "French_loaf",
    "n04522168": "vase",
    "n03764736": "milk_can",
    "n04118538": "rugby_ball",
    "n03887697": "paper_towel",
    "n13044778": "earthstar",
    "n03291819": "envelope",
    "n03770439": "miniskirt",
    "n03124170": "cowboy_hat",
    "n04487081": "trolleybus",
    "n03916031": "perfume",
    "n02808440": "bathtub",
    "n07697537": "hotdog",
    "n12985857": "coral_fungus",
    "n02917067": "bullet_train",
    "n03938244": "pillow",
    "n15075141": "toilet_tissue",
    "n02978881": "cassette",
    "n02966687": "carpenter's_kit",
    "n03633091": "ladle",
    "n13040303": "stinkhorn",
    "n03690938": "lotion",
    "n03476991": "hair_spray",
    "n02669723": "academic_gown",
    "n03220513": "dome",
    "n03127925": "crate",
    "n04584207": "wig",
    "n07880968": "burrito",
    "n03937543": "pill_bottle",
    "n03000247": "chain_mail",
    "n04418357": "theater_curtain",
    "n04590129": "window_shade",
    "n02795169": "barrel",
    "n04553703": "washbasin",
    "n02783161": "ballpoint",
    "n02802426": "basketball",
    "n02808304": "bath_towel",
    "n03124043": "cowboy_boot",
    "n03450230": "gown",
    "n04589890": "window_screen",
    "n12998815": "agaric",
    "n02992529": "cellular_telephone",
    "n03825788": "nipple",
    "n02790996": "barbell",
    "n03710193": "mailbox",
    "n03630383": "lab_coat",
    "n03347037": "fire_screen",
    "n03769881": "minibus",
    "n03871628": "packet",
    "n03733281": "maze",
    "n03976657": "pole",
    "n03535780": "horizontal_bar",
    "n04259630": "sombrero",
    "n03929855": "pickelhaube",
    "n04049303": "rain_barrel",
    "n04548362": "wallet",
    "n02979186": "cassette_player",
    "n06596364": "comic_book",
    "n03935335": "piggy_bank",
    "n06794110": "street_sign",
    "n02825657": "bell_cote",
    "n03388183": "fountain_pen",
    "n04591157": "Windsor_tie",
    "n04540053": "volleyball",
    "n03866082": "overskirt",
    "n04136333": "sarong",
    "n04026417": "purse",
    "n02865351": "bolo_tie",
    "n02834397": "bib",
    "n03888257": "parachute",
    "n04235860": "sleeping_bag",
    "n04404412": "television",
    "n04371430": "swimming_trunks",
    "n03733805": "measuring_cup",
    "n07920052": "espresso",
    "n07873807": "pizza",
    "n02895154": "breastplate",
    "n04204238": "shopping_basket",
    "n04597913": "wooden_spoon",
    "n04131690": "saltshaker",
    "n07836838": "chocolate_sauce",
    "n09835506": "ballplayer",
    "n03443371": "goblet",
    "n13037406": "gyromitra",
    "n04336792": "stretcher",
    "n04557648": "water_bottle",
    "n03187595": "dial_telephone",
    "n04254120": "soap_dispenser",
    "n03595614": "jersey",
    "n04146614": "school_bus",
    "n03598930": "jigsaw_puzzle",
    "n03958227": "plastic_bag",
    "n04069434": "reflex_camera",
    "n03188531": "diaper",
    "n02786058": "Band_Aid",
    "n07615774": "ice_lolly",
    "n04525038": "velvet",
    "n04409515": "tennis_ball",
    "n03424325": "gasmask",
    "n03223299": "doormat",
    "n03680355": "Loafer",
    "n07614500": "ice_cream",
    "n07695742": "pretzel",
    "n04033995": "quilt",
    "n03710721": "maillot",
    "n04392985": "tape_player",
    "n03047690": "clog",
    "n03584254": "iPod",
    "n13054560": "bolete",
    "n10565667": "scuba_diver",
    "n03950228": "pitcher",
    "n03729826": "matchstick",
    "n02837789": "bikini",
    "n04254777": "sock",
    "n02988304": "CD_player",
    "n03657121": "lens_cap",
    "n04417672": "thatch",
    "n04523525": "vault",
    "n02815834": "beaker",
    "n09229709": "bubble",
    "n07697313": "cheeseburger",
    "n03888605": "parallel_bars",
    "n03355925": "flagpole",
    "n03063599": "coffee_mug",
    "n04116512": "rubber_eraser",
    "n04325704": "stole",
    "n07831146": "carbonara",
    "n03255030": "dumbbell"
}
