import numpy as np
import functools


def genetics(
    global_ranking_indexes: np.ndarray[int],
    local_ranking_indexes: list[np.ndarray[int]],
    round_index: int,
    frequencies: list[int],
) -> tuple[np.ndarray[int], np.ndarray[int], np.ndarray[bool]]:
    # Check wich populations should evolve
    num_populations = len(frequencies)
    num_agents_per_population = len(local_ranking_indexes[0])
    num_agents = len(global_ranking_indexes)

    concerned_populations = [(round_index + 1) % freq == 0 for freq in frequencies]
    print("Concerned populations: ", concerned_populations, "\n")

    # Initialize the fathers with arange, meaning that if we don't do nothing
    # each agent will resume training unchanged
    fathers_hps = np.arange(num_agents)
    fathers_network = np.arange(num_agents)
    need_explore = np.zeros(num_agents, dtype=bool)

    # We don't need the rewards, but we can get an array such that:
    # inverse_ranking[i] > inverse_ranking[j] <=> rewards[i] > rewards[j]
    inverse_ranking = np.empty_like(global_ranking_indexes)
    for i in range(num_agents):
        inverse_ranking[global_ranking_indexes[i]] = num_agents - i

    # You can check that np.argsort(-inverse_ranking) = global_ranking

    def internal_exploit(local_ranking: np.ndarray[int]):
        # Within a population we get rid of the 25% worst agents
        # This function returns local indexes inside the population
        share = num_agents_per_population // 4
        fathers = np.arange(num_agents_per_population)
        fathers[local_ranking[-share:]] = local_ranking[:share]
        return fathers

    def local_to_global_index(agent_index: int, population_index: int):
        return agent_index + num_agents_per_population * population_index

    def global_to_local_index(global_index: int):
        # returns population_index, agent_index
        return divmod(global_index, num_agents_per_population)

    get_population = lambda x: global_to_local_index(x)[0]

    # Perform internal exploit for all concerned populations
    for population_index in range(num_populations):
        if concerned_populations[population_index]:
            fathers = internal_exploit(local_ranking_indexes[population_index])
            glob = functools.partial(
                local_to_global_index, population_index=population_index
            )
            for agent in range(num_agents_per_population):
                father = fathers[agent]
                if father != agent:
                    need_explore[glob(agent)] = True
                    fathers_hps[glob(agent)] = glob(father)
                    fathers_network[glob(agent)] = glob(father)

    # Then we handle the migration process
    def migration(
        local_ranking: np.ndarray[int],
        global_ranking: np.ndarray[int],
        population_index: int,
    ):
        # For each agent in the 3rd bracket apply him the migration process
        share = num_agents_per_population // 4
        migration_bracket = local_ranking[2 * share : 3 * share]

        # Need to go back to global indexes to compare with agents outside the population
        migration_bracket = [
            local_to_global_index(agent, population_index)
            for agent in migration_bracket
        ]

        # We need a list of the best agents that are not in the population
        external_ranking = [
            agent
            for agent in global_ranking
            if get_population(agent) != population_index
        ]

        def perform_transfer(agent: int, migrant: int):
            # We implement the transfer protocol of MF-PBT

            # In any case the network is inherited
            fathers_network[agent] = migrant

            # Two cases for the hyperparameters
            in_population = get_population(agent)
            out_population = get_population(migrant)

            if frequencies[in_population] < frequencies[out_population]:
                # In this case the agent exhibited greediness, we totally replace it
                fathers_hps[agent] = migrant
            elif frequencies[in_population] > frequencies[out_population]:
                # In this case we want to protect ourselves from greediness
                best_internal_agent = local_ranking[0]
                fathers_hps[agent] = local_to_global_index(
                    best_internal_agent, in_population
                )

        for agent in migration_bracket:
            # The candidate is the best external agent
            migrant = external_ranking[0]
            if inverse_ranking[agent] < inverse_ranking[migrant]:
                perform_transfer(agent, migrant)
                # We don't transfer twice the same agent
                external_ranking.pop(0)

    for population_index in range(num_populations):
        if concerned_populations[population_index]:
            migration(
                local_ranking_indexes[population_index],
                global_ranking_indexes,
                population_index,
            )

    return fathers_hps, fathers_network, need_explore
