import json
import jsmin
import os
import torch
import numpy as np
import math
from matplotlib import pyplot as plt


class Dict(dict):
    """
    Dictionary that allows to access per attributes and to except names from being loaded
    """
    def __init__(self, dictionary: dict = None):
        super(Dict, self).__init__()

        if dictionary is not None:
            self.load(dictionary)

    def __getattr__(self, item):
        try:
            return self[item] if item in self else getattr(super(Dict, self), item)
        except AttributeError:
            raise AttributeError(f'This dictionary has no attribute "{item}"')

    def load(self, dictionary: dict, name_list: list = None):
        """
        Loads a dictionary
        :param dictionary: Dictionary to be loaded
        :param name_list: List of names to be updated
        """
        for name in dictionary:
            data = dictionary[name]
            if name_list is None or name in name_list:
                if isinstance(data, dict):
                    if name in self:
                        self[name].load(data)
                    else:
                        self[name] = Dict(data)
                elif isinstance(data, list):
                    self[name] = list()
                    for item in data:
                        if isinstance(item, dict):
                            self[name].append(Dict(item))
                        else:
                            self[name].append(item)
                else:
                    self[name] = data

    def save(self, path):
        """
        Saves the dictionary into a json file
        :param path: Path of the json file
        """
        if not os.path.exists(path):
            os.makedirs(path)

        path = os.path.join(path, 'cfg.json')

        with open(path, 'w') as file:
            json.dump(self, file, indent=True)


class Configuration(Dict):
    """
    Configuration loaded from a json file
    """
    def __init__(self, path: str, default_path=None):
        super(Configuration, self).__init__()

        if default_path is not None:
            self.load(default_path)

        self.load(path)

    def load_model(self, path: str):
        self.load(path, name_list=["model"])

    def load(self, path: str, name_list: list = None):
        """
        Loads attributes from a json file
        :param path: Path of the json file
        :param name_list: List of names to be updated
        :return:
        """
        with open(path) as file:
            data = json.loads(jsmin.jsmin(file.read()))

            super(Configuration, self).load(data, name_list)


class Checkpoint():
    def __init__(self, dir, model, device):
        self.dir = dir
        if not os.path.exists(self.dir):
            os.makedirs(self.dir)
            
        self.model = model
        self.device = device
    
    def load(self, epoch):
        self.model.load_state_dict(torch.load("{}/{}.pt".format(self.dir, epoch)), map_location="cuda:{}".format(self.device))
    
    def save(self, epoch):
        torch.save(self.model.state_dict(),  "{}/{}.pt".format(self.dir, epoch))


def set_seed(num):
    torch.manual_seed(num)
    torch.cuda.manual_seed_all(num)
    np.random.seed(num)
    torch.backends.cudnn.deterministic = True


def get_num_params(model):
    total_num = 0
    for p in list(model.parameters()):
        num = p.size() + (2,) if p.is_complex() else p.size()
        total_num = total_num + math.prod(num)
    return total_num


class Null():
    def __init__(self, attr=None):
        self.attr = None
    
    def step(self):
        return


def logger(log_dir, np_list, str_list):
    for i in range(0, len(np_list)):
        np_list[i] = np.array(np_list[i])
        np.save(log_dir + "/" + str_list[i], np_list[i])
    
    for i in range(1, len(np_list)):
        plt.xlabel(str_list[0])
        plt.ylabel(str_list[i])
        plt.plot(np_list[0], np.log10(np_list[i]), label=str_list[i])
        plt.legend()
        plt.savefig(log_dir + "/" + str_list[i] + ".png")
        plt.clf()


def draw_1D(x, y, hold=True, x_label=None, y_label=None, filename=None):
    plt.plot(x, y["value"], label=y["label"])
    if not hold:
        plt.xlabel(x_label)
        plt.ylabel(y_label)
        plt.legend()
        plt.savefig(filename)
        plt.clf()


def draw_2D(x, y, u, x_label, y_label, u_label, filename=None):
    c = plt.pcolormesh(x, y, u, cmap='rainbow', shading='gouraud')
    plt.colorbar(c, label=u_label)
    plt.xlabel(x_label)
    plt.ylabel(y_label)
    plt.savefig(filename, dpi=300)
    plt.clf()
    

def show_Burgers(u, filename):
    u = torch.squeeze(u, -1)
    u = u.cpu().numpy()
    x = np.array([np.linspace(0, u.shape[0], u.shape[0])])
    x = np.repeat(x, [u.shape[1]], axis=0)
    y = np.array([np.linspace(0, u.shape[1], u.shape[1])])
    y = np.repeat(y, [u.shape[0]], axis=0).T
    draw_2D(x, y, u, "x", "t", "u", filename)

