# -*- coding: utf-8 -*-

import numpy as np
import pickle as pkl
import time
import os
import sys
from urllib import parse

from data import DataStreamer
from my_utils import Logit, MultiLogit
from network import Network
from system import SimulatedSystem, SimulatedSystemWithSymmetricGossiping

tolopogies = {'complete':'fc', 'ring':'cy', 'smallworld':'sm'}

# ARGUMENTS
data_name = 'pokerhand' # dataset
data_folder = '/home/dataset/' # data dir
save_folder = '../instantMA/' # save results dir

proc_delay = 1 # levels of processing delays
lr = .055 # learning rate
K = 100 # diameter of decision set
NODES = 32 # network size
topo = 'smallworld' # network topology
clf = MultiLogit # we use multilogit loss for multi-class classification

msg_delay = 30 # the level of message delays on all edges linking the reference node
msg_type = 'node'
asymcommu = False # True for our algorithm; False for the algorithm using symmetric gossiping communication

if not os.path.exists(save_folder):
    os.mkdir(save_folder)
if not os.path.exists('../network/'):
    os.mkdir('../network/')

network_name = topo + '-' + str(NODES)
tp = tolopogies[topo]
try:
    with open('../network/' + network_name + '.pkl', 'rb') as ff:
        network = pkl.load(ff)
except IOError:
    network = Network(NODES, topo)
    with open('../network/' + network_name + '.pkl', 'wb') as ff:
        pkl.dump(network, ff)

t0 = parse.unquote(time.asctime(time.localtime(time.time())))

if not os.path.exists(save_folder):
    os.mkdir(save_folder)
flog = open(save_folder + 'log-%s.txt'%(t0), 'w')

flog.write('Basic Info\n')
flog.write('--------------\n')
flog.write('Topo: %s\n'%(topo))
flog.write('Processing Delay: %.3f\n'%(proc_delay))
flog.write('Message Delay: %.3f\n'%(msg_delay))
flog.write('Asymmetric Communication: %s\n'%(asymcommu))
flog.write('--------------\n')

print('Topo: %s'%(topo))
print('Feedback Delay: %.3f'%(proc_delay))
print('Message Delay: %.3f'%(msg_delay))
print('Asymmetric Communication: %s'%(asymcommu))


flog.write('--------------\n')

stream = DataStreamer()
stream.load(data_name, data_folder)

if asymcommu:
    clf = SimulatedSystem(network)
else:
    clf = SimulatedSystemWithSymmetricGossiping(network)

avg_loss = clf.process(stream, record_round=2000, report_round=1e4, 
                       N=1e9, K=K, lr=lr, proc_delay=proc_delay, msg_delay=msg_delay, 
                       msg_type=msg_type, loss=MultiLogit, classes=10)
flog.write('%.3f, %.5f\n'%(lr, avg_loss))

with open(save_folder + '%.3f.pkl'%(lr), 'wb') as ff:
    pkl.dump(clf.RECORD, ff)

flog.close()


