## Setting

# Heterogeneous treatment effects [X1, X2] (taufun)
# Main effect [X1] (basefun)
# Confounding [X1] (psfun)
# External shift [X1, X2] (shiftfun)

## independent covariates 
Xfun1 <- function(n, d){
  matrix(runif(n * d), nrow = n, ncol = d)
}

## correlated covariates
rho <- 0.5
Xfun2 <- function(n, d){
  X <- matrix(rnorm(n * d), nrow = n, ncol = d)
  fac <- rnorm(n)
  X <- X * sqrt(1 - rho) + fac * sqrt(rho)
  pnorm(X)
}

# Equation (27) of Wager and Athey (2018)
basefun <- function(X){
  -1 * (X[, 1] - 0.5)
}

# Equation (29) of Wager and Athey (2018)
taufun <- function(X){
  2 / (1 + exp(-12 * (X[, 1] - 0.5))) * 2 / (1 + exp(-12 * (X[, 2] - 0.5)))
}

# Equation (27) of Wager and Athey (2018)
psfun <- function(X){ 
  (1 + pbeta(X[, 1], 2, 4)) / 4
}

# External shift
shiftfun <- function(X){
  1 / (1 + exp(3 * (X[, 1] - 0.5) + 3 * (X[, 2] - 0.5)))
}

## Generate data

gen_pop <- function(n, d,
                    Xfun, basefun, taufun, psfun, shiftfun){
  
  X <- Xfun(n, d)

  ps <- psfun(X) # propensity scores
  T <- rbinom(n, 1, ps) # treatment

  shift <- shiftfun(X) # shift
  shiftw <- (shift / (1 - shift)) # base weights
  
  tau <- taufun(X) # tau
  Y0l <- basefun(X)
  Y1l <- tau + basefun(X)
  Y0p <- 1 / (1 + exp(-Y0l))
  Y1p <- 1 / (1 + exp(-Y1l))
  taup <- Y1p - Y0p # tau p's
  
  Yl <- T * tau + basefun(X) # outcome logits
  Yp <- 1 / (1 + exp(-Yl)) # outcome p's
  Y <- rbinom(n, 1, Yp) # outcome class 

  return(data.frame(X, T, ps, Y0p, Y1p, Yl, Yp, Y, tau, taup, shift, shiftw))
}
