import random
import numpy as np
from collections import defaultdict
import networkx as nx

def efficient_bipartite_assignment(num_papers, num_reviewers, reviews_per_paper, max_reviews_per_reviewer, tolerance=1, randomness_level=0.3):
    """
    高效的二部图分配算法（带随机性）

    Parameters:
    - num_papers: 文章数量 (m)
    - num_reviewers: 审稿人数量 (n)
    - reviews_per_paper: 每篇文章需要的审稿数 (l)
    - max_reviews_per_reviewer: 每个审稿人最多审稿数 (r)
    - tolerance: 审稿人工作量的容忍差异 (q)
    - randomness_level: 随机性水平 [0.0-1.0], 0为完全贪心，1为完全随机

    Returns:
    - assignment_matrix: 分配矩阵 (num_papers x num_reviewers)
    """

    # 参数验证
    total_reviews_needed = num_papers * reviews_per_paper
    max_possible_reviews = num_reviewers * (max_reviews_per_reviewer + tolerance)

    if total_reviews_needed > max_possible_reviews:
        raise ValueError(f"Impossible assignment: need {total_reviews_needed} reviews, "
                        f"but can provide at most {max_possible_reviews}")

    # 使用贪心算法 + 随机化 + 局部调整
    assignment_matrix = np.zeros((num_papers, num_reviewers), dtype=int)
    reviewer_loads = np.zeros(num_reviewers, dtype=int)
    paper_assignments = np.zeros(num_papers, dtype=int)

    # 引入随机性：随机化文章处理顺序
    paper_order = list(range(num_papers))
    random.shuffle(paper_order)

    # Phase 1: 随机化贪心分配
    for paper_id in paper_order:
        # 为每篇文章找到reviews_per_paper个审稿人
        available_reviewers = []
        for reviewer_id in range(num_reviewers):
            # 检查是否已经分配给这篇文章
            if assignment_matrix[paper_id, reviewer_id] == 0:
                # 检查审稿人是否还有容量
                if reviewer_loads[reviewer_id] < max_reviews_per_reviewer + tolerance:
                    available_reviewers.append(reviewer_id)

        if len(available_reviewers) < reviews_per_paper:
            # 如果可用审稿人不足，按工作量排序
            available_reviewers.sort(key=lambda x: reviewer_loads[x])
        else:
            # 引入随机性：部分随机选择，部分贪心选择
            if random.random() < randomness_level:
                # 完全随机选择
                random.shuffle(available_reviewers)
            else:
                # 加权随机：工作量越少权重越高
                weights = []
                max_load = max(reviewer_loads[rid] for rid in available_reviewers) if available_reviewers else 0
                for rid in available_reviewers:
                    # 工作量越少，权重越高
                    weight = max_load - reviewer_loads[rid] + 1
                    weights.append(weight)

                # 基于权重的部分随机化
                selected_reviewers = []
                remaining_reviewers = available_reviewers.copy()
                remaining_weights = weights.copy()

                for _ in range(min(reviews_per_paper, len(available_reviewers))):
                    if random.random() < randomness_level:
                        # 随机选择
                        idx = random.randint(0, len(remaining_reviewers) - 1)
                    else:
                        # 按权重概率选择
                        total_weight = sum(remaining_weights)
                        if total_weight > 0:
                            rand_val = random.uniform(0, total_weight)
                            cumsum = 0
                            idx = 0
                            for i, w in enumerate(remaining_weights):
                                cumsum += w
                                if rand_val <= cumsum:
                                    idx = i
                                    break
                        else:
                            idx = 0

                    selected_reviewers.append(remaining_reviewers[idx])
                    remaining_reviewers.pop(idx)
                    remaining_weights.pop(idx)

                    if not remaining_reviewers:
                        break

                available_reviewers = selected_reviewers

        # 分配审稿人
        assigned_count = 0
        for reviewer_id in available_reviewers[:reviews_per_paper]:
            assignment_matrix[paper_id, reviewer_id] = 1
            reviewer_loads[reviewer_id] += 1
            assigned_count += 1

        paper_assignments[paper_id] = assigned_count

        # 如果分配不足，尝试从工作量较多的审稿人那里分配
        if assigned_count < reviews_per_paper:
            remaining = reviews_per_paper - assigned_count
            # 寻找还能承担额外工作的审稿人
            candidates = []
            for reviewer_id in range(num_reviewers):
                if (assignment_matrix[paper_id, reviewer_id] == 0 and
                    reviewer_loads[reviewer_id] < max_reviews_per_reviewer + tolerance):
                    candidates.append(reviewer_id)

            # 随机化候选人顺序
            random.shuffle(candidates)

            for reviewer_id in candidates:
                if remaining <= 0:
                    break
                assignment_matrix[paper_id, reviewer_id] = 1
                reviewer_loads[reviewer_id] += 1
                remaining -= 1

            paper_assignments[paper_id] = reviews_per_paper - remaining

    # Phase 2: 随机化负载均衡调整
    return balance_assignment_with_randomness(assignment_matrix, reviewer_loads, max_reviews_per_reviewer, tolerance, randomness_level)

def balance_assignment_with_randomness(assignment_matrix, reviewer_loads, target_load, tolerance, randomness_level=0.3):
    """
    带随机性的平衡审稿人工作量
    """
    num_papers, num_reviewers = assignment_matrix.shape
    max_iterations = 100  # 避免无限循环

    for iteration in range(max_iterations):
        # 找到工作量过高和过低的审稿人
        overloaded = []
        underloaded = []

        for reviewer_id in range(num_reviewers):
            load = reviewer_loads[reviewer_id]
            if load > target_load + tolerance:
                overloaded.append((reviewer_id, load))
            elif load < target_load - tolerance:
                underloaded.append((reviewer_id, load))

        if not overloaded or not underloaded:
            break  # 已经平衡

        # 随机化处理顺序
        random.shuffle(overloaded)
        random.shuffle(underloaded)

        # 尝试重新分配
        improved = False
        for over_reviewer, over_load in overloaded:
            for under_reviewer, under_load in underloaded:
                if over_load <= target_load + tolerance:
                    break  # 这个overloaded审稿人已经处理完

                # 寻找可以转移的文章
                transferable_papers = []
                for paper_id in range(num_papers):
                    if (assignment_matrix[paper_id, over_reviewer] == 1 and
                        assignment_matrix[paper_id, under_reviewer] == 0 and
                        under_load < target_load + tolerance):
                        transferable_papers.append(paper_id)

                # 随机选择要转移的文章
                if transferable_papers:
                    if random.random() < randomness_level:
                        # 随机选择
                        paper_id = random.choice(transferable_papers)
                    else:
                        # 选择第一个（保持一定确定性）
                        paper_id = transferable_papers[0]

                    # 执行转移
                    assignment_matrix[paper_id, over_reviewer] = 0
                    assignment_matrix[paper_id, under_reviewer] = 1
                    reviewer_loads[over_reviewer] -= 1
                    reviewer_loads[under_reviewer] += 1
                    over_load -= 1
                    under_load += 1
                    improved = True
                    break

        if not improved:
            break

    return assignment_matrix

def create_networkx_graph_from_matrix(assignment_matrix):
    """
    从分配矩阵创建NetworkX图对象，保持与原接口兼容
    """
    num_papers, num_reviewers = assignment_matrix.shape
    G = nx.Graph()

    # 添加节点
    paper_nodes = [f"L{i}" for i in range(num_papers)]
    reviewer_nodes = [f"R{i}" for i in range(num_reviewers)]

    G.add_nodes_from(paper_nodes, bipartite=0)
    G.add_nodes_from(reviewer_nodes, bipartite=1)

    # 添加边
    edges = []
    for paper_id in range(num_papers):
        for reviewer_id in range(num_reviewers):
            if assignment_matrix[paper_id, reviewer_id] == 1:
                edges.append((f"L{paper_id}", f"R{reviewer_id}"))

    G.add_edges_from(edges)
    return G

def efficient_bipartite_matching_with_tolerance(m, n, l, r, q, randomness_level=0.3):
    """
    高效版本的二部图匹配函数，替代原有的随机重试方法
    保持与原函数相同的接口，但增加随机性控制

    Parameters:
    - m: 文章数量
    - n: 审稿人数量
    - l: 每篇文章需要的审稿数
    - r: 每个审稿人的目标审稿数
    - q: 容忍度
    - randomness_level: 随机性水平 [0.0-1.0]
    """
    try:
        assignment_matrix = efficient_bipartite_assignment(m, n, l, r, q, randomness_level)
        return create_networkx_graph_from_matrix(assignment_matrix)
    except ValueError as e:
        print(f"Assignment failed: {e}")
        # 如果严格约束下无法分配，尝试放宽约束
        try:
            assignment_matrix = efficient_bipartite_assignment(m, n, l, r, q + 1, randomness_level)
            return create_networkx_graph_from_matrix(assignment_matrix)
        except ValueError:
            # 最后fallback到原始方法
            return fallback_random_assignment(m, n, l, r, q)

def weighted_random_choice(items, weights):
    """
    带权重的随机选择辅助函数
    """
    if not items or not weights:
        return None

    total_weight = sum(weights)
    if total_weight <= 0:
        return random.choice(items)

    rand_val = random.uniform(0, total_weight)
    cumsum = 0

    for item, weight in zip(items, weights):
        cumsum += weight
        if rand_val <= cumsum:
            return item

    return items[-1]  # fallback

def create_randomized_reviewer_selection(available_reviewers, reviewer_loads, reviews_per_paper, randomness_level):
    """
    创建随机化的审稿人选择策略

    Parameters:
    - available_reviewers: 可用审稿人列表
    - reviewer_loads: 审稿人当前工作量
    - reviews_per_paper: 需要的审稿人数量
    - randomness_level: 随机性水平

    Returns:
    - selected_reviewers: 选中的审稿人列表
    """
    if len(available_reviewers) <= reviews_per_paper:
        return available_reviewers

    if randomness_level >= 0.8:
        # 高随机性：主要随机选择
        selected = random.sample(available_reviewers, reviews_per_paper)
        return selected

    elif randomness_level <= 0.2:
        # 低随机性：主要贪心选择
        sorted_reviewers = sorted(available_reviewers, key=lambda x: reviewer_loads[x])
        return sorted_reviewers[:reviews_per_paper]

    else:
        # 中等随机性：混合策略
        # 一部分贪心选择，一部分随机选择
        greedy_count = int(reviews_per_paper * (1 - randomness_level))
        random_count = reviews_per_paper - greedy_count

        # 贪心部分
        sorted_reviewers = sorted(available_reviewers, key=lambda x: reviewer_loads[x])
        selected = sorted_reviewers[:greedy_count]

        # 随机部分
        remaining = [r for r in available_reviewers if r not in selected]
        if remaining and random_count > 0:
            additional = random.sample(remaining, min(random_count, len(remaining)))
            selected.extend(additional)

        return selected

def fallback_random_assignment(m, n, l, r, q, max_attempts=1000):
    """
    当确定性方法失败时的备选随机方法，但限制尝试次数
    """
    for attempt in range(max_attempts):
        G = nx.Graph()
        left_nodes = [f"L{i}" for i in range(m)]
        right_nodes = [f"R{i}" for i in range(n)]

        G.add_nodes_from(left_nodes, bipartite=0)
        G.add_nodes_from(right_nodes, bipartite=1)

        edges = []
        for left in left_nodes:
            if len(right_nodes) >= l:
                connections = random.sample(right_nodes, l)
            else:
                connections = right_nodes
            for right in connections:
                edges.append((left, right))

        random.shuffle(edges)
        G.add_edges_from(edges[:m * l])

        # 检查约束
        valid = True
        for node in left_nodes:
            if abs(G.degree(node) - l) > q:
                valid = False
                break

        if valid:
            for node in right_nodes:
                if abs(G.degree(node) - r) > q:
                    valid = False
                    break

        if valid:
            return G

    # 如果随机方法也失败，创建一个基本可行的分配
    return create_basic_feasible_assignment(m, n, l, r, q)

def create_basic_feasible_assignment(m, n, l, r, q):
    """
    创建一个基本可行分配作为最后的备选方案
    """
    G = nx.Graph()
    left_nodes = [f"L{i}" for i in range(m)]
    right_nodes = [f"R{i}" for i in range(n)]

    G.add_nodes_from(left_nodes, bipartite=0)
    G.add_nodes_from(right_nodes, bipartite=1)

    # 简单的循环分配
    edges = []
    reviewer_idx = 0

    for paper_idx in range(m):
        for _ in range(l):
            edges.append((f"L{paper_idx}", f"R{reviewer_idx % n}"))
            reviewer_idx += 1

    G.add_edges_from(edges)
    return G

def validate_assignment(assignment_matrix, reviews_per_paper, max_reviews_per_reviewer, tolerance):
    """
    验证分配是否满足约束
    """
    num_papers, num_reviewers = assignment_matrix.shape

    # 检查每篇文章的审稿数
    paper_reviews = assignment_matrix.sum(axis=1)
    for paper_id, count in enumerate(paper_reviews):
        if abs(count - reviews_per_paper) > tolerance:
            print(f"Paper {paper_id} has {count} reviews, expected {reviews_per_paper}±{tolerance}")
            return False

    # 检查每个审稿人的工作量
    reviewer_loads = assignment_matrix.sum(axis=0)
    for reviewer_id, load in enumerate(reviewer_loads):
        if abs(load - max_reviews_per_reviewer) > tolerance:
            print(f"Reviewer {reviewer_id} has {load} assignments, expected {max_reviews_per_reviewer}±{tolerance}")
            return False

    return True

# 示例和测试
if __name__ == "__main__":
    # 测试各种随机性水平
    test_configs = [
        (20, 20, 6, 6, 1),  # 基本配置
        (10, 15, 3, 2, 1),  # 不平衡配置
        (50, 30, 3, 5, 2),  # 大规模配置
    ]

    randomness_levels = [0.0, 0.3, 0.6, 1.0]

    for m, n, l, r, q in test_configs:
        print(f"\nTesting configuration: {m} papers, {n} reviewers, {l} reviews/paper, {r} max/reviewer, tolerance {q}")

        for rand_level in randomness_levels:
            print(f"  Randomness level: {rand_level}")

            try:
                # 测试新方法
                assignment_matrix = efficient_bipartite_assignment(m, n, l, r, q, rand_level)

                if validate_assignment(assignment_matrix, l, r, q):
                    print("    ✓ Assignment valid")
                    reviewer_loads = assignment_matrix.sum(axis=0)
                    print(f"    Reviewer load variance: {np.var(reviewer_loads):.2f}")
                else:
                    print("    ✗ Assignment invalid")

            except ValueError as e:
                print(f"    ✗ Assignment failed: {e}")